/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "dlgramjet.h"

#include "wndmain.h"
#include <QMessageBox>


#define FETCH_DOUBLE(SET, FIELD) \
    if (!fRamjet) return; \
    try { \
      doFetchComments(); \
      fRamjet -> SET(dblFromLineEdit(FIELD())); \
      doRefresh(); \
    } catch (std::invalid_argument &exc) { \
      showException(exc); \
    }
// FETCH_DOUBLE


#define FETCH_DOUBLE_RECALC(SET, FIELD, RULE) \
    if (!fRamjet) return; \
      doFetchComments(); \
      fRamjet -> SET(dblFromLineEdit(FIELD()), RULE()); \
      doRefresh(); \
    try { \
    } catch(std::invalid_argument &exc) { \
      showException(exc); \
    }
// FETCH_DOUBLE_RECALC


namespace GerFuSp {

  const char* DlgRamjet::TITLE_EMPTY = "Edit Ramjet";
  const char* DlgRamjet::TITLE_FULL  = "%s '%s' - Edit Ramjet";

  const int DlgRamjet::EDIT_WIDTH = 64;

  DlgRamjet::DlgRamjet() {
    fRamjet = NULL;
    fBoxMain = NULL;
    fGridData = NULL;
    fLbName = NULL;
    fLnName = NULL;
    fLbReaction = NULL;
    fCbReaction = NULL;
    fLbOutput = NULL;
    fLnOutput = NULL;
    fLbShareGIM = NULL;
    fLnShareGIM = NULL;
    fLbUsedShare = NULL;
    fLnUsedShare = NULL;
    fLbShare = NULL;
    fLnShare = NULL;
    fLbEfficiency = NULL;
    fLnEfficiency = NULL;
    fLbConsumption = NULL;
    fLnConsumption = NULL;
    fLbStructure = NULL;
    fLnStructure = NULL;
    fLbIntakeRadius = NULL;
    fLnIntakeRadius = NULL;
    fLbPower = NULL;
    fLnPower = NULL;
    fLbDissipation = NULL;
    fLnDissipation = NULL;
    fLbThrust = NULL;
    fLnThrust = NULL;
    fLbExhVel_c = NULL;
    fLnExhVel_c = NULL;
    fLbExhVel_km_s = NULL;
    fLnExhVel_km_s = NULL;
    fLbTrvSpd_c = NULL;
    fLnTrvSpd_c = NULL;
    fLbTrvSpd_km_s = NULL;
    fLnTrvSpd_km_s = NULL;
    fLbAcceleration = NULL;
    fLnAcceleration = NULL;
    fRdRecalc_w_p = NULL;
    fRdRecalc_w_eta = NULL;
    fRdRecalc_P_p = NULL;
    fRdRecalc_P_eta = NULL;
    fRdRecalc_P_mu = NULL;
    fBoxComments = NULL;
    fLbComments = NULL;
    fTextComments = NULL;
    fBoxCtrl = NULL;
    fBtApply = NULL;
    fBtCancel = NULL;
    fBtReload = NULL;
    fBtSave = NULL;
    fBtAbort = NULL;
    fBtOK = NULL;
    fGrpRecalcExhVel = NULL;
    fGrpRecalcPower = NULL;
    fPath_w_p = NULL;
    fPath_w_eta = NULL;
    fPath_P_p = NULL;
    fPath_P_eta = NULL;
    fPath_P_mu = NULL;
    fFtRecalc = QRadioButton().font();
    fFtRecalc.setPointSize(fFtRecalc.pointSize() * 3/4);
    fRefreshing = false;
    setLayout(getBoxMain());
    setWindowTitle(tr(TITLE_EMPTY));
    setWindowIcon(WndMain::getTheWindow() -> windowIcon());
  } // DlgRamjet::DlgRamjet

  void DlgRamjet::doAbort() {
    doReload();
    close();
  } // DlgRamjet::doAbort

  void DlgRamjet::doApply() {
    doFetchComments();
    focusNextChild();
    doRefresh();
  } // DlgRamjet::doApply

  void DlgRamjet::doFetchAcceleration() {
    FETCH_DOUBLE(setAcceleration_g, getLnAcceleration);
  } // DlgRamjet::doFetchAcceleration

  void DlgRamjet::doFetchComments() {
    if (fRefreshing || !fRamjet) {
      return;
    } // if
    fRamjet -> setComments(getTextComments() -> toPlainText());
  } // DlgRamjet::doFetchComments

  void DlgRamjet::doFetchConsumption() {
    FETCH_DOUBLE(setConsumption_kg_s, getLnConsumption);
  } // DlgRamjet::doFetchConsumption

  void DlgRamjet::doFetchDissipation() {
    FETCH_DOUBLE_RECALC(setDissipation_W, getLnDissipation, getRecalcPower);
  } // DlgRamjet::doFetchDissipation

  void DlgRamjet::doFetchEfficiency() {
    FETCH_DOUBLE(setEfficiency, getLnEfficiency);
  } // DlgRamjet::doFetchEfficiency

  void DlgRamjet::doFetchExhVel_c() {
    FETCH_DOUBLE_RECALC(setExhaustVelocity_c, getLnExhVel_c, getRecalcExhVel);
  } // DlgRamjet::doFetchExhVel_c

  void DlgRamjet::doFetchExhVel_km_s() {
    FETCH_DOUBLE_RECALC(setExhaustVelocity_km_s, getLnExhVel_km_s,
      getRecalcExhVel);
  } // DlgRamjet::doFetchExhVel_km_s

  void DlgRamjet::doFetchIntakeRadius() {
    FETCH_DOUBLE(setIntakeRadius_km, getLnIntakeRadius);
  } // DlgRamjet::doFetchIntakeRadius

  void DlgRamjet::doFetchName() {
    if (!fRamjet) {
      return;
    } // if
    doFetchComments();
    fRamjet -> setName(getLnName() -> text().trimmed());
    doRefresh();
  } // DlgRamjet::doFetchName

  void DlgRamjet::doFetchPower() {
    FETCH_DOUBLE_RECALC(setPower_W, getLnPower, getRecalcPower);
  } // DlgRamjet::doFetchPower

  void DlgRamjet::doFetchShare() {
    FETCH_DOUBLE(setReactiveShare, getLnShare);
  } // DlgRamjet::doFetchShare

  void DlgRamjet::doFetchStructure() {
    FETCH_DOUBLE(setStructure_t, getLnStructure);
  } // DlgRamjet::doFetchStructure

  void DlgRamjet::doFetchThrust() {
    FETCH_DOUBLE(setThrust_N, getLnThrust);
  } // DlgRamjet::doFetchThrust

  void DlgRamjet::doFetchTrvSpd_c() {
    FETCH_DOUBLE_RECALC(setTravelSpeed_c, getLnTrvSpd_c, getRecalcExhVel);
  } // DlgRamjet::doFetchTrvSpd_c

  void DlgRamjet::doFetchTrvSpd_km_s() {
    FETCH_DOUBLE_RECALC(setTravelSpeed_km_s, getLnTrvSpd_km_s,
      getRecalcExhVel);
  } // DlgRamjet::doFetchTrvSpd_km_s

  void DlgRamjet::doFetchUsedShare() {
    FETCH_DOUBLE(setUsedShare, getLnUsedShare);
  } // DlgRamjet::doFetchUsedShare

  void DlgRamjet::doOK() {
    doSave();
    close();
  } // DlgRamjet::doOK

  void DlgRamjet::doReactionChanged() {
    QComboBox *combo = getCbReaction();
    int i = combo -> currentIndex();
    QVariant ID = combo -> itemData(i);
    Reaction *reaction = (Reaction*) Reaction::getTheContainer() ->
      getData(ID);
    if (fRamjet && !fRefreshing) {
      fRamjet -> setReaction(reaction);
      doRefresh();
    } // if
    if (reaction) {
      initLineEditF(getLnOutput(), reaction -> getOutput(), "%0.6f");
      getLnOutput() -> setReadOnly(true);
      initLineEditF(getLnShareGIM(), reaction -> getShareGIM(), "%0.6f");
      getLnShareGIM() -> setReadOnly(true);
    } else {
      getLnOutput() -> setText("");
    } // if
  } // DlgRamjet::doReactionChanged

  void DlgRamjet::doRefresh() {
    fRefreshing = true;
    try {
      reInitCbReaction();
      if (fRamjet) {
        resetTitle();
        initLineEdit(getLnName(), fRamjet -> getName());
	gotoComboItem(getCbReaction(), fRamjet -> getReaction());
	initLineEditF(getLnUsedShare(), fRamjet -> getUsedShare(), "%0.4f");
	initLineEditF(getLnShare(), fRamjet -> getReactiveShare(), "%0.6f");
	initLineEditF(getLnEfficiency(), fRamjet -> getEfficiency(), "%0.4f",
	  1.0e-5);
	initLineEditF(getLnConsumption(), fRamjet -> getConsumption_kg_s(),
	  "%0.2f", 1.0e-2, 1.0e+4);
	initLineEditF(getLnStructure(), fRamjet -> getStructure_t(), "%0.0f",
	  1.0, 1.0e+6);
	initLineEditF(getLnIntakeRadius(), fRamjet -> getIntakeRadius_km(),
	  "%0.0f", 1.0, 1.0e+6);
	initLineEditF(getLnPower(), fRamjet -> getPower_W(), "%0.2e");
	initLineEditF(getLnDissipation(), fRamjet -> getDissipation_W(),
	  "%0.2e");
	initLineEditF(getLnThrust(), fRamjet -> getThrust_N(), "%0.2e");
	initLineEditF(getLnExhVel_c(), fRamjet -> getExhaustVelocity_c(),
	  "%0.4f", 1.0e-4);
	initLineEditF(getLnExhVel_km_s(), fRamjet -> getExhaustVelocity_km_s(),
	  "%0.1f");
	initLineEditF(getLnTrvSpd_c(), fRamjet -> getTravelSpeed_c(),
	  "%0.4f", 1.0e-4);
	initLineEditF(getLnTrvSpd_km_s(), fRamjet -> getTravelSpeed_km_s(),
	  "%0.1f");
	initLineEditF(getLnAcceleration(), fRamjet -> getAcceleration_g(),
	  "%0.3f", 1.0e-3);
	getTextComments() -> setPlainText(fRamjet -> getComments());
	getTextComments() -> setReadOnly(false);
      } else {
        initLineEdit(getLnName(), "", true);
	gotoComboItem(getCbReaction(), NULL);
	initLineEdit(getLnUsedShare(), "", true);
	initLineEdit(getLnEfficiency(), "", true);
	initLineEdit(getLnConsumption(), "", true);
	initLineEdit(getLnStructure(), "", true);
	initLineEdit(getLnIntakeRadius(), "", true);
	initLineEdit(getLnPower(), "", true);
	initLineEdit(getLnDissipation(), "", true);
	initLineEdit(getLnThrust(), "", true);
	initLineEdit(getLnExhVel_c(), "", true);
	initLineEdit(getLnExhVel_km_s(), "", true);
	initLineEdit(getLnTrvSpd_c(), "", true);
	initLineEdit(getLnTrvSpd_km_s(), "", true);
	initLineEdit(getLnAcceleration(), "", true);
	getTextComments() -> setPlainText("");
	getTextComments() -> setReadOnly(true);
      } // if
      fRefreshing = false;
    } catch (std::exception &aExc) {
      fRefreshing = false;
    } // try
  } // DlgRamjet::doRefresh

  void DlgRamjet::doReload() {
    if (fRamjet) {
      fRamjet -> read();
    } // if
    doRefresh();
  } // DlgRamjet::doReload

  void DlgRamjet::doSave() {
    if (fRamjet) {
      doApply();
      fRamjet -> write();
      WndMain::getTheWindow() -> putShip(fRamjet);
    } // if
  } // DlgRamjet::doSave

  QVBoxLayout* DlgRamjet::getBoxComments() {
    if (!fBoxComments) {
      fBoxComments = new QVBoxLayout();
      fBoxComments -> addWidget(getLbComments());
      fBoxComments -> addWidget(getTextComments());
    } // if
    return fBoxComments;
  } // DlgRamjet::getBoxComments

  QHBoxLayout* DlgRamjet::getBoxCtrl() {
    if (!fBoxCtrl) {
      fBoxCtrl = new QHBoxLayout();
      fBoxCtrl -> addWidget(getBtApply());
      fBoxCtrl -> addWidget(getBtCancel());
      fBoxCtrl -> addWidget(getBtReload());
      fBoxCtrl -> addWidget(getBtSave());
      fBoxCtrl -> addStretch();
      fBoxCtrl -> addWidget(getBtAbort());
      fBoxCtrl -> addWidget(getBtOK());
    } // if
    return fBoxCtrl;
  } // DlgRamjet::getBoxCtrl

  QVBoxLayout* DlgRamjet::getBoxMain() {
    if (!fBoxMain) {
      fBoxMain = new QVBoxLayout();
      fBoxMain -> addLayout(getGridData());
      fBoxMain -> addLayout(getBoxComments());
      fBoxMain -> addLayout(getBoxCtrl());
    } // if
    return fBoxMain;
  } // DlgRamjet::getBoxMain

  QPushButton* DlgRamjet::getBtAbort() {
    if (!fBtAbort) {
      fBtAbort = new QPushButton(tr("Abort"));
      fBtAbort -> setShortcut(tr("Shift+Esc"));
      connect(fBtAbort, SIGNAL(clicked()), this, SLOT(doAbort()));
    } // if
    return fBtAbort;
  } // DlgRamjet::getBtAbort

  QPushButton* DlgRamjet::getBtApply() {
    if (!fBtApply) {
      fBtApply = new QPushButton(tr("Apply"));
      connect(fBtApply, SIGNAL(clicked()), this, SLOT(doApply()));
    } // if
    return fBtApply;
  } // DlgRamjet::getBtApply

  QPushButton* DlgRamjet::getBtCancel() {
    if (!fBtCancel) {
      fBtCancel = new QPushButton(tr("Cancel"));
      fBtCancel -> setShortcut(tr("Esc"));
      connect(fBtCancel, SIGNAL(clicked()), this, SLOT(doRefresh()));
    } // if
    return fBtCancel;
  } // DlgRamjet::getBtCancel

  QPushButton* DlgRamjet::getBtOK() {
    if (!fBtOK) {
      fBtOK = new QPushButton(tr("OK"));
      fBtOK -> setShortcut(tr("Shift+Return"));
      connect(fBtOK, SIGNAL(clicked()), this, SLOT(doOK()));
    } // if
    return fBtOK;
  } // DlgRamjet::getBtOK

  QPushButton* DlgRamjet::getBtReload() {
    if (!fBtReload) {
      fBtReload = new QPushButton(tr("Reload"));
      fBtReload -> setShortcut(tr("Ctrl+L"));
      connect(fBtReload, SIGNAL(clicked()), this, SLOT(doReload()));
    } // if
    return fBtReload;
  } // DlgRamjet::getBtReload

  QPushButton* DlgRamjet::getBtSave() {
    if (!fBtSave) {
      fBtSave = new QPushButton(tr("Save"));
      fBtSave -> setShortcut(tr("Ctrl+S"));
      connect(fBtSave, SIGNAL(clicked()), this, SLOT(doSave()));
    } // if
    return fBtSave;
  } // DlgRamjet::getBtSave

  QComboBox* DlgRamjet::getCbReaction() {
    if (!fCbReaction) {
      fCbReaction = new QComboBox();
      connect(fCbReaction, SIGNAL(currentIndexChanged(int)), this,
        SLOT(doReactionChanged()));
    } // if
    return fCbReaction;
  } // DlgRamjet::getCbReaction

  QGridLayout* DlgRamjet::getGridData() {
    if (!fGridData) {
      fGridData = new QGridLayout();
      /* Top block: */
      fGridData -> addWidget(getLbName(), 0, 0);
      fGridData -> addWidget(getLnName(), 0, 1, 1, 9);
      fGridData -> addWidget(getLbReaction(), 1, 3);
      fGridData -> addWidget(getCbReaction(), 1, 4, 1, 3);
      fGridData -> addWidget(getLbOutput(), 1, 7, 1, 2);
      fGridData -> addWidget(getLnOutput(), 1, 9);
      fGridData -> addWidget(getLbShareGIM(), 2, 0, 1, 9);
      fGridData -> addWidget(getLnShareGIM(), 2, 9);
      /* Main input column: */
      fGridData -> addWidget(getLbUsedShare(), 3, 3);
      fGridData -> addWidget(getLnUsedShare(), 3, 4);
      fGridData -> addWidget(getLbEfficiency(), 4, 3);
      fGridData -> addWidget(getLnEfficiency(), 4, 4);
      fGridData -> addWidget(getLbConsumption(), 5, 3);
      fGridData -> addWidget(getLnConsumption(), 5, 4);
      fGridData -> addWidget(getLbStructure(), 6, 3);
      fGridData -> addWidget(getLnStructure(), 6, 4);
      fGridData -> addWidget(getLbIntakeRadius(), 7, 3);
      fGridData -> addWidget(getLnIntakeRadius(), 7, 4);
      /* Thrust and power column: */
      fGridData -> addWidget(getLbPower(), 6, 0);
      fGridData -> addWidget(getLnPower(), 6, 1);
      fGridData -> addWidget(getLbDissipation(), 7, 0);
      fGridData -> addWidget(getLnDissipation(), 7, 1);
      fGridData -> addWidget(getLbThrust(), 8, 0);
      fGridData -> addWidget(getLnThrust(), 8, 1);
      /* Velocity column: */
      fGridData -> addWidget(getLbShare(), 3, 6, 1, 3);
      fGridData -> addWidget(getLnShare(), 3, 9);
      fGridData -> addWidget(getLbExhVel_c(), 5, 6);
      fGridData -> addWidget(getLnExhVel_c(), 5, 7);
      fGridData -> addWidget(getLbExhVel_km_s(), 5, 8);
      fGridData -> addWidget(getLnExhVel_km_s(), 5, 9);
      fGridData -> addWidget(getLbTrvSpd_c(), 7, 6);
      fGridData -> addWidget(getLnTrvSpd_c(), 7, 7);
      fGridData -> addWidget(getLbTrvSpd_km_s(), 7, 8);
      fGridData -> addWidget(getLnTrvSpd_km_s(), 7, 9);
      fGridData -> addWidget(getLbAcceleration(), 8, 6);
      fGridData -> addWidget(getLnAcceleration(), 8, 7);
      /* Recalc rules: */
      fGridData -> addWidget(getRdRecalc_w_p(), 3, 5);
      fGridData -> addWidget(getRdRecalc_w_eta(), 4, 5);
      fGridData -> addWidget(getRdRecalc_P_p(), 3, 2);
      fGridData -> addWidget(getRdRecalc_P_eta(), 4, 2);
      fGridData -> addWidget(getRdRecalc_P_mu(), 5, 2);
    } // if
    return fGridData;
  } // DlgRamjet::getGridData

  QButtonGroup* DlgRamjet::getGrpRecalcExhVel() {
    if (!fGrpRecalcExhVel) {
      fGrpRecalcExhVel = new QButtonGroup(this);
      fGrpRecalcExhVel -> setExclusive(true);
    } // if
    return fGrpRecalcExhVel;
  } // DlgRamjet::getGrpRecalcExhVel

  QButtonGroup* DlgRamjet::getGrpRecalcPower() {
    if (!fGrpRecalcPower) {
      fGrpRecalcPower = new QButtonGroup(this);
      fGrpRecalcPower -> setExclusive(true);
    } // if
    return fGrpRecalcPower;
  } // DlgRamjet::getGrpRecalcPower

  QLabel* DlgRamjet::getLbAcceleration() {
    if (!fLbAcceleration) {
      fLbAcceleration = new QLabel(tr("Max. &Acceleration [g]:"));
      fLbAcceleration -> setAlignment(Qt::AlignRight);
      fLbAcceleration -> setBuddy(getLnAcceleration());
    } // if
    return fLbAcceleration;
  } // DlgRamjet::getLbAcceleration

  QLabel* DlgRamjet::getLbComments() {
    if (!fLbComments) {
      fLbComments = new QLabel(tr("Co&mments:"));
      fLbComments -> setBuddy(getTextComments());
    } // if
    return fLbComments;
  } // DlgRamjet::getLbComments

  QLabel* DlgRamjet::getLbConsumption() {
    if (!fLbConsumption) {
      fLbConsumption = new QLabel(tr("Max. Consum&ption [kg/s]:"));
      fLbConsumption -> setAlignment(Qt::AlignRight);
      fLbConsumption -> setBuddy(getLnConsumption());
    } // if
    return fLbConsumption;
  } // DlgRamjet::getLbConsumption

  QLabel* DlgRamjet::getLbDissipation() {
    if (!fLbDissipation) {
      fLbDissipation = new QLabel(tr("&Dissipation [W]:"));
      fLbDissipation -> setAlignment(Qt::AlignRight);
      fLbDissipation -> setBuddy(getLnDissipation());
    } // if
    return fLbDissipation;
  } // DlgRamjet::getLbDissipation

  QLabel* DlgRamjet::getLbEfficiency() {
    if (!fLbEfficiency) {
      fLbEfficiency = new QLabel(tr("Efficienc&y:"));
      fLbEfficiency -> setAlignment(Qt::AlignRight);
      fLbEfficiency -> setBuddy(getLnEfficiency());
    } // if
    return fLbEfficiency;
  } // DlgRamjet::getLbEfficiency

  QLabel* DlgRamjet::getLbExhVel_c() {
    if (!fLbExhVel_c) {
      fLbExhVel_c = new QLabel(tr("E&xhaust velocity [c]:"));
      fLbExhVel_c -> setAlignment(Qt::AlignRight);
      fLbExhVel_c -> setBuddy(getLnExhVel_c());
    } // if
    return fLbExhVel_c;
  } // DlgRamjet::getLbExhVel_c

  QLabel* DlgRamjet::getLbExhVel_km_s() {
    if (!fLbExhVel_km_s) {
      fLbExhVel_km_s = new QLabel(tr("= [km/s]"));
      fLbExhVel_km_s -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbExhVel_km_s;
  } // DlgRamjet::getLbExhVel_km_s

  QLabel* DlgRamjet::getLbIntakeRadius() {
    if (!fLbIntakeRadius) {
      fLbIntakeRadius = new QLabel(tr("&Intake Radius [km]:"));
      fLbIntakeRadius -> setAlignment(Qt::AlignRight);
      fLbIntakeRadius -> setBuddy(getLnIntakeRadius());
    } // if
    return fLbIntakeRadius;
  } // DlgRamjet::getLbIntakeRadius

  QLabel* DlgRamjet::getLbName() {
    if (!fLbName) {
      fLbName = new QLabel(tr("&Name:"));
      fLbName -> setAlignment(Qt::AlignRight);
      fLbName -> setBuddy(getLnName());
    } // if
    return fLbName;
  } // DlgRamjet::getLbName

  QLabel* DlgRamjet::getLbOutput() {
    if (!fLbOutput) {
      fLbOutput = new QLabel(tr("Output:"));
      fLbOutput -> setAlignment(Qt::AlignRight);
    } // if
    return fLbOutput;
  } // DlgRamjet::getLbOutput

  QLabel* DlgRamjet::getLbPower() {
    if (!fLbPower) {
      fLbPower = new QLabel(tr("Po&wer [W]:"));
      fLbPower -> setAlignment(Qt::AlignRight);
      fLbPower -> setBuddy(getLnPower());
    } // if
    return fLbPower;
  } // DlgRamjet::getLbPower

  QLabel* DlgRamjet::getLbReaction() {
    if (!fLbReaction) {
      fLbReaction = new QLabel(tr("&Reaction:"));
      fLbReaction -> setAlignment(Qt::AlignRight);
      fLbReaction -> setBuddy(getCbReaction());
    } // if
    return fLbReaction;
  } // DlgRamjet::getLbReaction

  QLabel* DlgRamjet::getLbShare() {
    if (!fLbShare) {
      fLbShare = new QLabel(tr("Total reactive &share:"));
      fLbShare -> setAlignment(Qt::AlignRight);
      fLbShare -> setBuddy(getLnShare());
    } // if
    return fLbShare;
  } // DlgRamjet::getLbShare

  QLabel* DlgRamjet::getLbShareGIM() {
    if (!fLbShareGIM) {
      fLbShareGIM = new QLabel(
        tr("Share of reactants in the galactic interstellar medium:"));
      fLbShareGIM -> setAlignment(Qt::AlignRight);
    } // if
    return fLbShareGIM;
  } // DlgRamjet::GetLbShareGIM

  QLabel* DlgRamjet::getLbStructure() {
    if (!fLbStructure) {
      fLbStructure = new QLabel(tr("Str&ucture mass [t]:"));
      fLbStructure -> setAlignment(Qt::AlignRight);
      fLbStructure -> setBuddy(getLnStructure());
    } // if
    return fLbStructure;
  } // DlgRamjet::getLbStructure

  QLabel* DlgRamjet::getLbThrust() {
    if (!fLbThrust) {
      fLbThrust = new QLabel(tr("T&hrust [N]:"));
      fLbThrust -> setAlignment(Qt::AlignRight);
      fLbThrust -> setBuddy(getLnThrust());
    } // if
    return fLbThrust;
  } // DlgRamjet::getLbThrust

  QLabel* DlgRamjet::getLbTrvSpd_c() {
    if (!fLbTrvSpd_c) {
      fLbTrvSpd_c = new QLabel(tr("Tra&vel speed [c]:"));
      fLbTrvSpd_c -> setAlignment(Qt::AlignRight);
      fLbTrvSpd_c -> setBuddy(getLnTrvSpd_c());
    } // if
    return fLbTrvSpd_c;
  } // DlgRamjet::getLbTrvSpd_c

  QLabel* DlgRamjet::getLbTrvSpd_km_s() {
    if (!fLbTrvSpd_km_s) {
      fLbTrvSpd_km_s = new QLabel(tr("= [km/s]"));
      fLbTrvSpd_km_s -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbTrvSpd_km_s;
  } // DlgRamjet::getLbTrvSpd_km_s

  QLabel* DlgRamjet::getLbUsedShare() {
    if (!fLbUsedShare) {
      fLbUsedShare = new QLabel(tr("Used fue&l share:"));
      fLbUsedShare -> setAlignment(Qt::AlignRight);
      fLbUsedShare -> setBuddy(getLnUsedShare());
    } // if
    return fLbUsedShare;
  } // DlgRamjet::getLbUsedShare

  QLineEdit* DlgRamjet::getLnAcceleration() {
    if (!fLnAcceleration) {
      fLnAcceleration = new QLineEdit();
      fLnAcceleration -> setMinimumWidth(EDIT_WIDTH);
      fLnAcceleration -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnAcceleration, SIGNAL(editingFinished()), this,
        SLOT(doFetchAcceleration()));
    } // if
    return fLnAcceleration;
  } // DlgRamjet::getLnAcceleration

  QLineEdit* DlgRamjet::getLnConsumption() {
    if (!fLnConsumption) {
      fLnConsumption = new QLineEdit();
      fLnConsumption -> setMinimumWidth(EDIT_WIDTH);
      fLnConsumption -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnConsumption, SIGNAL(editingFinished()), this,
        SLOT(doFetchConsumption()));
    } // if
    return fLnConsumption;
  } // DlgRamjet::getLnConsumption

  QLineEdit* DlgRamjet::getLnDissipation() {
    if (!fLnDissipation) {
      fLnDissipation = new QLineEdit();
      fLnDissipation -> setMinimumWidth(EDIT_WIDTH);
      fLnDissipation -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnDissipation, SIGNAL(editingFinished()), this,
        SLOT(doFetchDissipation()));
    } // if
    return fLnDissipation;
  } // DlgRamjet::getLnDissipation

  QLineEdit* DlgRamjet::getLnEfficiency() {
    if (!fLnEfficiency) {
      fLnEfficiency = new QLineEdit();
      fLnEfficiency -> setMinimumWidth(EDIT_WIDTH);
      fLnEfficiency -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnEfficiency, SIGNAL(editingFinished()), this,
        SLOT(doFetchEfficiency()));
    } // if
    return fLnEfficiency;
  } // DlgRamjet::getLnEfficiency

  QLineEdit* DlgRamjet::getLnExhVel_c() {
    if (!fLnExhVel_c) {
      fLnExhVel_c = new QLineEdit();
      fLnExhVel_c -> setMinimumWidth(EDIT_WIDTH);
      fLnExhVel_c -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnExhVel_c, SIGNAL(editingFinished()), this, SLOT(doFetchExhVel_c()));
    } // if
    return fLnExhVel_c;
  } // DlgRamjet::getLnExhVel_c

  QLineEdit* DlgRamjet::getLnExhVel_km_s() {
    if (!fLnExhVel_km_s) {
      fLnExhVel_km_s = new QLineEdit();
      fLnExhVel_km_s -> setMinimumWidth(EDIT_WIDTH);
      fLnExhVel_km_s -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnExhVel_km_s, SIGNAL(editingFinished()), this,
        SLOT(doFetchExhVel_km_s()));
    } // if
    return fLnExhVel_km_s;
  } // DlgRamjet::getLnExhVel_km_s

  QLineEdit* DlgRamjet::getLnIntakeRadius() {
    if (!fLnIntakeRadius) {
      fLnIntakeRadius = new QLineEdit();
      fLnIntakeRadius -> setMinimumWidth(EDIT_WIDTH);
      fLnIntakeRadius -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnIntakeRadius, SIGNAL(editingFinished()), this,
        SLOT(doFetchIntakeRadius()));
    } // if
    return fLnIntakeRadius;
  } // DlgRamjet::getLnIntakeRadius

  QLineEdit* DlgRamjet::getLnName() {
    if (!fLnName) {
      fLnName = new QLineEdit();
      connect(fLnName, SIGNAL(editingFinished()), this, SLOT(doFetchName()));
    } // if
    return fLnName;
  } // DlgRamjet::getLnName

  QLineEdit* DlgRamjet::getLnOutput() {
    if (!fLnOutput) {
      fLnOutput = new QLineEdit();
      fLnOutput -> setMinimumWidth(EDIT_WIDTH);
      fLnOutput -> setMaximumWidth(EDIT_WIDTH);
      fLnOutput -> setReadOnly(true);
    } // if
    return fLnOutput;
  } // DlgRamjet::getLnOutput

  QLineEdit* DlgRamjet::getLnPower() {
    if (!fLnPower) {
      fLnPower = new QLineEdit();
      fLnPower -> setMinimumWidth(EDIT_WIDTH);
      fLnPower -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnPower, SIGNAL(editingFinished()), this, SLOT(doFetchPower()));
    } // if
    return fLnPower;
  } // DlgRamjet::getLnPower

  QLineEdit* DlgRamjet::getLnShare() {
    if (!fLnShare) {
      fLnShare = new QLineEdit();
      fLnShare -> setMinimumWidth(EDIT_WIDTH);
      fLnShare -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnShare, SIGNAL(editingFinished()), this, SLOT(doFetchShare()));
    } // if
    return fLnShare;
  } // DlgRamjet::getLnShare

  QLineEdit* DlgRamjet::getLnShareGIM() {
    if (!fLnShareGIM) {
      fLnShareGIM = new QLineEdit();
      fLnShareGIM -> setMinimumWidth(EDIT_WIDTH);
      fLnShareGIM -> setMaximumWidth(EDIT_WIDTH);
      fLnShareGIM -> setReadOnly(true);
    } // if
    return fLnShareGIM;
  } // DlgRamjet::getLnShareGIM

  QLineEdit* DlgRamjet::getLnStructure() {
    if (!fLnStructure) {
      fLnStructure = new QLineEdit();
      fLnStructure -> setMinimumWidth(EDIT_WIDTH);
      fLnStructure -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnStructure, SIGNAL(editingFinished()), this,
        SLOT(doFetchStructure()));
    } // if
    return fLnStructure;
  } // DlgRamjet::getLnStructure

  QLineEdit* DlgRamjet::getLnThrust() {
    if (!fLnThrust) {
      fLnThrust = new QLineEdit();
      fLnThrust -> setMinimumWidth(EDIT_WIDTH);
      fLnThrust -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnThrust, SIGNAL(editingFinished()), this, SLOT(doFetchThrust()));
    } // if
    return fLnThrust;
  } // DlgRamjet::getLnThrust

  QLineEdit* DlgRamjet::getLnTrvSpd_c() {
    if (!fLnTrvSpd_c) {
      fLnTrvSpd_c = new QLineEdit();
      fLnTrvSpd_c -> setMinimumWidth(EDIT_WIDTH);
      fLnTrvSpd_c -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnTrvSpd_c, SIGNAL(editingFinished()), this, SLOT(doFetchTrvSpd_c()));
    } // if
    return fLnTrvSpd_c;
  } // DlgRamjet::getLnTrvSpd_c

  QLineEdit* DlgRamjet::getLnTrvSpd_km_s() {
    if (!fLnTrvSpd_km_s) {
      fLnTrvSpd_km_s = new QLineEdit();
      fLnTrvSpd_km_s -> setMinimumWidth(EDIT_WIDTH);
      fLnTrvSpd_km_s -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnTrvSpd_km_s, SIGNAL(editingFinished()), this,
        SLOT(doFetchTrvSpd_km_s()));
    } // if
    return fLnTrvSpd_km_s;
  } // DlgRamjet::getLnTrvSpd_km_s

  QLineEdit* DlgRamjet::getLnUsedShare() {
    if (!fLnUsedShare) {
      fLnUsedShare = new QLineEdit();
      fLnUsedShare -> setMinimumWidth(EDIT_WIDTH);
      fLnUsedShare -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnUsedShare, SIGNAL(editingFinished()), this,
        SLOT(doFetchUsedShare()));
    } // if
    return fLnUsedShare;
  } // DlgRamjet::getLnUsedShare

  RecalcPath* DlgRamjet::getPath_P_eta() {
    if (!fPath_P_eta) {
      fPath_P_eta = new RecalcPath(getGridData(), 7, 1, 4, 4, Qt::blue,
        Qt::Vertical);
    } // if
    return fPath_P_eta;
  } // DlgRamjet::getPath_P_eta

  RecalcPath* DlgRamjet::getPath_P_mu() {
    if (!fPath_P_mu) {
      fPath_P_mu = new RecalcPath(getGridData(), 7, 1, 5, 4, Qt::blue,
        Qt::Vertical);
    } // if
    return fPath_P_mu;
  } // DlgRamjet::getPath_P_mu

  RecalcPath* DlgRamjet::getPath_P_p() {
    if (!fPath_P_p) {
      fPath_P_p = new RecalcPath(getGridData(), 7, 1, 3, 4, Qt::blue,
        Qt::Vertical);
    } // if
    return fPath_P_p;
  } // DlgRamjet::getPath_P_p

  RecalcPath* DlgRamjet::getPath_w_eta() {
    if (!fPath_w_eta) {
      fPath_w_eta = new RecalcPath(getGridData(), 7, 7, 4, 4, Qt::red,
        Qt::Vertical);
    } // if
    return fPath_w_eta;
  } // DlgRamjet::getPath_w_eta

  RecalcPath* DlgRamjet::getPath_w_p() {
    if (!fPath_w_p) {
      fPath_w_p = new RecalcPath(getGridData(), 7, 7, 3, 4, Qt::red,
        Qt::Vertical);
    } // if
    return fPath_w_p;
  } // DlgRamjet::getPath_w_p

  Ramjet* DlgRamjet::getRamjet() {
    return fRamjet;
  } // DlgRamjet::getRamjet

  QRadioButton* DlgRamjet::getRdRecalc_P_eta() {
    if (!fRdRecalc_P_eta) {
      fRdRecalc_P_eta = new QRadioButton("&2");
      fRdRecalc_P_eta -> setFont(fFtRecalc);
      getGrpRecalcPower() -> addButton(fRdRecalc_P_eta);
      connect(fRdRecalc_P_eta, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_P_eta;
  } // DlgRamjet::getRdRecalc_P_eta

  QRadioButton* DlgRamjet::getRdRecalc_P_mu() {
    if (!fRdRecalc_P_mu) {
      fRdRecalc_P_mu = new QRadioButton("&3");
      fRdRecalc_P_mu -> setFont(fFtRecalc);
      getGrpRecalcPower() -> addButton(fRdRecalc_P_mu);
      connect(fRdRecalc_P_mu, SIGNAL(toggled(bool)), this, SLOT(repaint()));
      fRdRecalc_P_mu -> setChecked(true);
    } // if
    return fRdRecalc_P_mu;
  } // DlgRamjet::getRdRecalc_P_mu

  QRadioButton* DlgRamjet::getRdRecalc_P_p() {
    if (!fRdRecalc_P_p) {
      fRdRecalc_P_p = new QRadioButton("&1");
      fRdRecalc_P_p -> setFont(fFtRecalc);
      getGrpRecalcPower() -> addButton(fRdRecalc_P_p);
      connect(fRdRecalc_P_p, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_P_p;
  } // DlgRamjet::getRdRecalc_P_p

  QRadioButton* DlgRamjet::getRdRecalc_w_eta() {
    if (!fRdRecalc_w_eta) {
      fRdRecalc_w_eta = new QRadioButton("&5");
      fRdRecalc_w_eta -> setFont(fFtRecalc);
      getGrpRecalcExhVel() -> addButton(fRdRecalc_w_eta);
      connect(fRdRecalc_w_eta, SIGNAL(toggled(bool)), this, SLOT(repaint()));
      fRdRecalc_w_eta -> setChecked(true);
    } // if
    return fRdRecalc_w_eta;
  } // DlgRamjet::getRdRecalc_w_eta

  QRadioButton* DlgRamjet::getRdRecalc_w_p() {
    if (!fRdRecalc_w_p) {
      fRdRecalc_w_p = new QRadioButton("&4");
      fRdRecalc_w_p -> setFont(fFtRecalc);
      getGrpRecalcExhVel() -> addButton(fRdRecalc_w_p);
      connect(fRdRecalc_w_p, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_w_p;
  } // DlgRamjet::getRdRecalc_w_p

  Ship::RecalcRule DlgRamjet::getRecalcExhVel() {
    if (getRdRecalc_w_p() -> isChecked()) {
      return Ship::rrByShare;
    } else {
      return Ship::rrByEfficiency;
    } // if
  } // DlgRamjet::getRecalcExhVel

  Ship::RecalcRule DlgRamjet::getRecalcPower() {
    if (getRdRecalc_P_p() -> isChecked()) {
      return Ship::rrByShare;
    } else if (getRdRecalc_P_eta() -> isChecked()) {
      return Ship::rrByEfficiency;
    } else {
      return Ship::rrByFuel;
    } // if
  } // DlgRamjet::getRecalcPower

  QTextEdit* DlgRamjet::getTextComments() {
    if (!fTextComments) {
      fTextComments = new QTextEdit();
    } // if
    return fTextComments;
  } // DlgRamjet::getTextComments

  void DlgRamjet::paintEvent(QPaintEvent *aEvent) {
    QDialog::paintEvent(aEvent);
    QPainter painter(this);
    Ship::RecalcRule rr_w = getRecalcExhVel();
    Ship::RecalcRule rr_P = getRecalcPower();
    getPath_w_p() -> draw(painter, rr_w == Ship::rrByShare);
    getPath_w_eta() -> draw(painter, rr_w == Ship::rrByEfficiency);
    getPath_P_p() -> draw(painter, rr_P == Ship::rrByShare);
    getPath_P_eta() -> draw(painter, rr_P == Ship::rrByEfficiency);
    getPath_P_mu() -> draw(painter, rr_P == Ship::rrByFuel);
  } // DlgRamjet::paintEvent

  void DlgRamjet::reInitCbReaction() {
    GerFuSp::reInitCbReaction(getCbReaction(), tr("<none>"));
  } // DlgRamjet::reInitCbReaction

  void DlgRamjet::resetTitle() {
    if (fRamjet) {
      setWindowTitle(QString().sprintf(qPrintable(tr(TITLE_FULL)),
        qPrintable(IDToStr(fRamjet -> getID())),
	qPrintable(fRamjet -> getName())));
    } else {
      setWindowTitle(tr(TITLE_EMPTY));
    } // if
  } // DlgRamjet::resetTitle

  void DlgRamjet::setRamjet(Ramjet *aValue) {
    fRamjet = aValue;
    resetTitle();
    doRefresh();
    QLineEdit *edit = getLnName();
    edit -> selectAll();
    edit -> setFocus();
  } // DlgRamjet::setRamjet

  void DlgRamjet::showException(const std::exception &aException) {
    QMessageBox::information(this, tr("Error"), tr(aException.what()));
  } // DlgRamjet::showException

} // GerFuSp
