/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "dlgrocket.h"

#include "gerfuspbase.h"
#include "reaction.h"
#include "wndmain.h"
#include <QColor>
#include <QMessageBox>


#define FETCH_DOUBLE(SET, FIELD) \
    if (!fRocket) return; \
    try { \
      doFetchComments(); \
      fRocket -> SET(dblFromLineEdit(FIELD())); \
      doRefresh(); \
    } catch (std::invalid_argument &exc) { \
      showException(exc); \
    }
// FETCH_DOUBLE


#define FETCH_DOUBLE_RECALC(SET, FIELD, RULE) \
    if (!fRocket) return; \
    try { \
      doFetchComments(); \
      fRocket -> SET(dblFromLineEdit(FIELD()), RULE()); \
      doRefresh(); \
    } catch (std::invalid_argument &exc) { \
      showException(exc); \
    }
// FETCH_DOUBLE_RECALC


namespace GerFuSp {

  const char* DlgRocket::TITLE_EMPTY = "Edit Rocket";
  const char* DlgRocket::TITLE_FULL  = "%s '%s' - Edit Rocket";

  const int DlgRocket::EDIT_WIDTH = 64;

  DlgRocket::DlgRocket() {
    fRocket = NULL;
    fBoxMain = NULL;
    fGridData = NULL;
    fLbName = NULL;
    fLnName = NULL;
    fLbReaction = NULL;
    fCbReaction = NULL;
    fLbOutput = NULL;
    fLnOutput = NULL;
    fLbIce = NULL;
    fCbIce = NULL;
    fLbShareReactive = NULL;
    fLnShareReactive = NULL;
    fLbShare = NULL;
    fLnShare = NULL;
    fLbAddPropPerFuel = NULL;
    fLnAddPropPerFuel = NULL;
    fLbEfficiency = NULL;
    fLnEfficiency = NULL;
    fLbStructure = NULL;
    fLnStructure = NULL;
    fLbPower = NULL;
    fLnPower = NULL;
    fLbDissipation = NULL;
    fLnDissipation = NULL;
    fLbThrust = NULL;
    fLnThrust = NULL;
    fLbConsumption = NULL;
    fLnConsumption = NULL;
    fLbStartEmptyRatio = NULL;
    fLnStartEmptyRatio = NULL;
    fLbStartingMass = NULL;
    fLnStartingMass = NULL;
    fLbAccelFull = NULL;
    fLnAccelFull = NULL;
    fLbAccelEmpty = NULL;
    fLnAccelEmpty = NULL;
    fLbFuelMass = NULL;
    fLnFuelMass = NULL;
    fLbIceQuantity = NULL;
    fLnIceQuantity = NULL;
    fLbExhVel_c = NULL;
    fLnExhVel_c = NULL;
    fLbExhVel_km_s = NULL;
    fLnExhVel_km_s = NULL;
    fLbTrvSpd_c = NULL;
    fLnTrvSpd_c = NULL;
    fLbTrvSpd_km_s = NULL;
    fLnTrvSpd_km_s = NULL;
    fRdRecalc_w_p = NULL;
    fRdRecalc_w_eta = NULL;
    fRdRecalc_v_w = NULL;
    fRdRecalc_v_R = NULL;
    fRdRecalc_P_p = NULL;
    fRdRecalc_P_eta = NULL;
    fRdRecalc_P_mu = NULL;
    fLbInertial = NULL;
    fLbOnBoard = NULL;
    fLbTotalFiringTime = NULL;
    fLnTotalFiringTimeInertial = NULL;
    fLnTotalFiringTimeOnBoard = NULL;
    fLbAccelTime = NULL;
    fLnAccelTimeInertial = NULL;
    fLnAccelTimeOnBoard = NULL;
    fLbDecelTime = NULL;
    fLnDecelTimeInertial = NULL;
    fLnDecelTimeOnBoard = NULL;
    fLbTotalFiringDistance_ly = NULL;
    fLnTotalFiringDistance_ly = NULL;
    fLbTotalFiringDistance_km = NULL;
    fLnTotalFiringDistance_km = NULL;
    fLbAccelDistance_ly = NULL;
    fLnAccelDistance_ly = NULL;
    fLbAccelDistance_km = NULL;
    fLnAccelDistance_km = NULL;
    fLbDecelDistance_ly = NULL;
    fLnDecelDistance_ly = NULL;
    fLbDecelDistance_km = NULL;
    fLnDecelDistance_km = NULL;
    fBoxComments = NULL;
    fLbComments = NULL;
    fTextComments = NULL;
    fBoxCtrl = NULL;
    fBtApply = NULL;
    fBtCancel = NULL;
    fBtReload = NULL;
    fBtSave = NULL;
    fBtAbort = NULL;
    fBtOK = NULL;
    fGrpRecalcExhVel = NULL;
    fGrpRecalcTrvSpd = NULL;
    fGrpRecalcPower = NULL;
    fPath_w_p = NULL;
    fPath_w_eta = NULL;
    fPath_v_w = NULL;
    fPath_v_R = NULL;
    fPath_P_p = NULL;
    fPath_P_eta = NULL;
    fPath_P_mu = NULL;
    fFtRecalc = QRadioButton().font();
    fFtRecalc.setPointSize(fFtRecalc.pointSize() * 3/4);
    fRefreshing = false;
    setLayout(getBoxMain());
    setWindowTitle(tr(TITLE_EMPTY));
    setWindowIcon(WndMain::getTheWindow() -> windowIcon());
  } // DlgRocket::DlgRocket

  void DlgRocket::doAbort() {
    doReload();
    close();
  } // DlgRocket::doAbort

  void DlgRocket::doApply() {
    doFetchComments();
    focusNextChild();
    doRefresh();
  } // DlgRocket::doApply

  void DlgRocket::doFetchAccelDistance_ly() {
    FETCH_DOUBLE(setAccelDistance_ly, getLnAccelDistance_ly);
  } // DlgRocket::doFetchAccelDistance_ly

  void DlgRocket::doFetchAccelDistance_km() {
    FETCH_DOUBLE(setAccelDistance_km, getLnAccelDistance_km);
  } // DlgRocket::doFetchAccelDistance_km

  void DlgRocket::doFetchAccelEmpty() {
    FETCH_DOUBLE(setAccelEmpty_g, getLnAccelEmpty);
  } // DlgRocket::doFetchAccelEmpty

  void DlgRocket::doFetchAccelFull() {
    FETCH_DOUBLE(setAccelFull_g, getLnAccelFull);
  } // DlgRocket::doFetchAccelFull

  void DlgRocket::doFetchAddPropPerFuel() {
    FETCH_DOUBLE(setAddPropPerFuel, getLnAddPropPerFuel);
  } // DlgRocket::doFetchAddPropPerFuel

  void DlgRocket::doFetchComments() {
    if (fRefreshing || !fRocket) {
      return;
    } // if
    fRocket -> setComments(getTextComments() -> toPlainText());
  } // DlgRocket::doFetchComments

  void DlgRocket::doFetchConsumption() {
    FETCH_DOUBLE(setConsumption, getLnConsumption);
  } // DlgRocket::doFetchConsumption

  void DlgRocket::doFetchDecelDistance_ly() {
    FETCH_DOUBLE(setDecelDistance_ly, getLnDecelDistance_ly);
  } // DlgRocket::doFetchDecelDistance_ly

  void DlgRocket::doFetchDecelDistance_km() {
    FETCH_DOUBLE(setDecelDistance_km, getLnDecelDistance_km);
  } // DlgRocket::doFetchDecelDistance_km

  void DlgRocket::doFetchDissipation() {
    FETCH_DOUBLE_RECALC(setDissipation_W, getLnDissipation, getRecalcPower);
  } // DlgRocket::doFetchDissipation

  void DlgRocket::doFetchEfficiency() {
    FETCH_DOUBLE(setEfficiency, getLnEfficiency);
  } // DlgRocket::doFetchEfficiency

  void DlgRocket::doFetchExhVel_c() {
    FETCH_DOUBLE_RECALC(setExhaustVelocity_c, getLnExhVel_c, getRecalcExhVel);
  } // DlgRocket::doFetchExhVel_c

  void DlgRocket::doFetchExhVel_km_s() {
    FETCH_DOUBLE_RECALC(setExhaustVelocity_km_s, getLnExhVel_km_s,
      getRecalcExhVel);
  } // DlgRocket::doFetchExhVel_km_s

  void DlgRocket::doFetchFuelMass() {
    FETCH_DOUBLE(setFuelMass_t, getLnFuelMass);
  } // DlgRocket::doFetchFuelMass

  void DlgRocket::doFetchIceQuantity() {
    FETCH_DOUBLE(setIceDiameter_km, getLnIceQuantity);
  } // DlgRocket::doFetchIceQuantity

  void DlgRocket::doFetchName() {
    if (!fRocket) {
      return;
    } // if
    doFetchComments();
    fRocket -> setName(getLnName() -> text().trimmed());
    resetTitle();
    doRefresh();
  } // DlgRocket::doFetchName

  void DlgRocket::doFetchPower() {
    FETCH_DOUBLE_RECALC(setPower_W, getLnPower, getRecalcPower);
  } // DlgRocket::doFetchPower

  void DlgRocket::doFetchReactiveShare() {
    FETCH_DOUBLE(setReactiveShare, getLnShare);
  } // DlgRocket::doFetchReactiveShare

  void DlgRocket::doFetchStartEmptyRatio() {
    FETCH_DOUBLE(setRatioFullEmpty, getLnStartEmptyRatio);
  } // DlgRocket::doFetchStartEmptyRatio

  void DlgRocket::doFetchStartingMass() {
    FETCH_DOUBLE(setStartingMass_t, getLnStartingMass);
  } // DlgRocket::doFetchStartingMass

  void DlgRocket::doFetchStructure() {
    FETCH_DOUBLE(setStructure_t, getLnStructure);
  } // DlgRocket::doFetchStructure

  void DlgRocket::doFetchThrust() {
    FETCH_DOUBLE(setThrust_N, getLnThrust);
  } // DlgRocket::doFetchThrust

  void DlgRocket::doFetchTotalFiringDistance_ly() {
    FETCH_DOUBLE(setTotalFiringDistance_ly, getLnTotalFiringDistance_ly);
  } // DlgRocket::doFetchTotalFiringDistance_ly

  void DlgRocket::doFetchTotalFiringDistance_km() {
    FETCH_DOUBLE(setTotalFiringDistance_km, getLnTotalFiringDistance_km);
  } // DlgRocket::doFetchTotalFiringDistance_km

  void DlgRocket::doFetchTrvSpd_c() {
    FETCH_DOUBLE_RECALC(setTravelSpeed_c, getLnTrvSpd_c, getRecalcTrvSpd);
  } // DlgRocket::doFetchTrvSpd_c

  void DlgRocket::doFetchTrvSpd_km_s() {
    FETCH_DOUBLE_RECALC(setTravelSpeed_km_s, getLnTrvSpd_km_s,
      getRecalcTrvSpd);
  } // DlgRocket::doFetchTrvSpd_km_s

  void DlgRocket::doIceChanged() {
    QComboBox *combo = getCbIce();
    int i = combo -> currentIndex();
    QVariant ID = combo -> itemData(i);
    Ice *ice = (Ice*) Ice::getTheContainer() -> getData(ID);
    if (fRocket && !fRefreshing) {
      fRocket -> setIce(ice);
      doRefresh();
    } // if
    if (ice) {
      getLnShareReactive() -> setText(QString().sprintf("%0.6f",
        ice -> getReactiveShare()));
    } else {
      getLnShareReactive() -> setText("");
    } // if
  } // DlgRocket::doIceChanged

  void DlgRocket::doOK() {
    doSave();
    close();
  } // DlgRocket::doOK

  void DlgRocket::doReactionChanged() {
    QComboBox *combo = getCbReaction();
    int i = combo -> currentIndex();
    QVariant ID = combo -> itemData(i);
    Reaction *reaction = (Reaction*) Reaction::getTheContainer() ->
      getData(ID);
    if (fRocket && !fRefreshing) {
      fRocket -> setReaction(reaction);
      doRefresh();
    } // if
    if (reaction) {
      initLineEditF(getLnOutput(), reaction -> getOutput(), "%0.6f");
      getLnOutput() -> setReadOnly(true);
    } else {
      getLnOutput() -> setText("");
    } // if
  } // DlgRocket::doReactionChanged

  void DlgRocket::doRefresh() {
    fRefreshing = true;
    try {
      reInitCbReaction();
      reInitCbIce();
      if (fRocket) {
	initLineEdit(getLnName(), fRocket -> getName());
	gotoComboItem(getCbReaction(), fRocket -> getReaction());
	gotoComboItem(getCbIce(), fRocket -> getIce());
	initLineEditF(getLnAddPropPerFuel(), fRocket -> getAddPropPerFuel(),
	  "%0.2f", 0.0, 1.0e+5);
	initLineEditF(getLnEfficiency(), fRocket -> getEfficiency(),
	  "%0.4f", 1.0e-5);
	initLineEditF(getLnConsumption(), fRocket -> getConsumption_kg_s(),
	  "%0.2e");
	initLineEditF(getLnStructure(), fRocket -> getStructure_t(),
	  "%0.0f", 1.0, 1.0e+6);
	initLineEditF(getLnStartEmptyRatio(), fRocket -> getRatioFullEmpty(),
	  "%0.2f", 0.0, 1.0e+3);
	initLineEditF(getLnStartingMass(), fRocket -> getStartingMass_t(),
	  "%0.0f", 1.0, 1.0e+6);
	initLineEditF(getLnFuelMass(), fRocket -> getFuelMass_t(),
	  "%0.0f", 1.0, 1.0e+6);
	initLineEditF(getLnPower(), fRocket -> getPower_W(), "%0.2e");
	initLineEditF(getLnDissipation(), fRocket -> getDissipation_W(),
	  "%0.2e");
	initLineEditF(getLnThrust(), fRocket -> getThrust_N(), "%0.2e");
	initLineEditF(getLnShare(), fRocket -> getReactiveShare(), "%0.6f");
	initLineEditF(getLnExhVel_c(), fRocket -> getExhaustVelocity_c(),
	  "%0.4f", 1.0e-4);
	initLineEditF(getLnExhVel_km_s(), fRocket -> getExhaustVelocity_km_s(),
	  "%0.1f");
	initLineEditF(getLnTrvSpd_c(), fRocket -> getTravelSpeed_c(),
	  "%0.4f", 1.0e-4);
	initLineEditF(getLnTrvSpd_km_s(), fRocket -> getTravelSpeed_km_s(),
	  "%0.1f");
	initLineEditF(getLnAccelFull(), fRocket -> getAccelFull_g(),
	  "%0.3f", 1.0e-3);
	initLineEditF(getLnAccelEmpty(), fRocket -> getAccelEmpty_g(),
	  "%0.3f", 1.0e-3, 1.0e+4);
	if (fRocket -> getIce()) {
	  initLineEditF(getLnIceQuantity(), fRocket -> getIceDiameter_km(),
	    "%0.2f", 1.0e-2);
	} else {
	  initLineEdit(getLnIceQuantity(), "", true);
	} // if
	initLineEdit(getLnTotalFiringTimeInertial(),
	  DurationToStr(fRocket -> getTotalFiringTimeInertial()), true);
	initLineEdit(getLnTotalFiringTimeOnBoard(),
	  DurationToStr(fRocket -> getTotalFiringTimeOnBoard()), true);
	initLineEdit(getLnAccelTimeInertial(),
	  DurationToStr(fRocket -> getAccelTimeInertial()), true);
	initLineEdit(getLnAccelTimeOnBoard(),
	  DurationToStr(fRocket -> getAccelTimeOnBoard()), true);
	initLineEdit(getLnDecelTimeInertial(),
	  DurationToStr(fRocket -> getDecelTimeInertial()), true);
	initLineEdit(getLnDecelTimeOnBoard(),
	  DurationToStr(fRocket -> getDecelTimeOnBoard()), true);
	initLineEditF(getLnTotalFiringDistance_ly(),
	  fRocket -> getTotalFiringDistance_ly(), "%0.3f", 1.0e-2, 1.0e+4);
	initLineEditF(getLnTotalFiringDistance_km(),
	  fRocket -> getTotalFiringDistance_km(), "%0.2e");
	initLineEditF(getLnAccelDistance_ly(),
	  fRocket -> getAccelDistance_ly(), "%0.3f", 1.0e-2, 1.0e+4);
	initLineEditF(getLnAccelDistance_km(),
	  fRocket -> getAccelDistance_km(), "%0.2e");
	initLineEditF(getLnDecelDistance_ly(),
	  fRocket -> getDecelDistance_ly(), "%0.3f", 1.0e-2, 1.0e+4);
	initLineEditF(getLnDecelDistance_km(),
	  fRocket -> getDecelDistance_km(), "%0.2e");
	getTextComments() -> setPlainText(fRocket -> getComments());
	getTextComments() -> setReadOnly(false);
      } else {
	initLineEdit(getLnName(), "", true);
	gotoComboItem(getCbReaction(), NULL);
	gotoComboItem(getCbIce(), NULL);
	initLineEdit(getLnAddPropPerFuel(), "", true);
	initLineEdit(getLnEfficiency(), "", true);
	initLineEdit(getLnConsumption(), "", true);
	initLineEdit(getLnStructure(), "", true);
	initLineEdit(getLnStartEmptyRatio(), "", true);
	initLineEdit(getLnStartingMass(), "", true);
	initLineEdit(getLnFuelMass(), "", true);
	initLineEdit(getLnPower(), "", true);
	initLineEdit(getLnDissipation(), "", true);
	initLineEdit(getLnThrust(), "", true);
	initLineEdit(getLnShare(), "", true);
	initLineEdit(getLnExhVel_c(), "", true);
	initLineEdit(getLnExhVel_km_s(), "", true);
	initLineEdit(getLnTrvSpd_c(), "", true);
	initLineEdit(getLnTrvSpd_km_s(), "", true);
	initLineEdit(getLnAccelFull(), "", true);
	initLineEdit(getLnAccelEmpty(), "", true);
	initLineEdit(getLnIceQuantity(), "", true);
	initLineEdit(getLnTotalFiringTimeInertial(), "", true);
	initLineEdit(getLnTotalFiringTimeOnBoard(), "", true);
	initLineEdit(getLnTotalFiringDistance_ly(), "", true);
	initLineEdit(getLnTotalFiringDistance_km(), "", true);
	initLineEdit(getLnAccelTimeInertial(), "", true);
	initLineEdit(getLnAccelTimeOnBoard(), "", true);
	initLineEdit(getLnAccelDistance_ly(), "", true);
	initLineEdit(getLnAccelDistance_km(), "", true);
	initLineEdit(getLnDecelTimeInertial(), "", true);
	initLineEdit(getLnDecelTimeOnBoard(), "", true);
	initLineEdit(getLnDecelDistance_ly(), "", true);
	initLineEdit(getLnDecelDistance_km(), "", true);
	getTextComments() -> setPlainText("");
	getTextComments() -> setReadOnly(true);
      } // if
      fRefreshing = false;
    } catch (std::exception &aExc) {
      fRefreshing = false;
    } // try
  } // DlgRocket::doRefresh

  void DlgRocket::doReload() {
    if (fRocket) {
      fRocket -> read();
    } // if
    doRefresh();
  } // DlgRocket::doReload

  void DlgRocket::doSave() {
    if (fRocket) {
      doApply();
      fRocket -> write();
      WndMain::getTheWindow() -> putShip(fRocket);
    } // if
  } // DlgRocket::doSave

  QVBoxLayout* DlgRocket::getBoxMain() {
    if (!fBoxMain) {
      fBoxMain = new QVBoxLayout();
      fBoxMain -> addLayout(getGridData());
      fBoxMain -> addLayout(getBoxComments());
      fBoxMain -> addLayout(getBoxCtrl());
    } // if
    return fBoxMain;
  } // DlgRocket::getBoxMain

  QVBoxLayout* DlgRocket::getBoxComments() {
    if (!fBoxComments) {
      fBoxComments = new QVBoxLayout();
      fBoxComments -> addWidget(getLbComments());
      fBoxComments -> addWidget(getTextComments());
    } // if
    return fBoxComments;
  } // DlgRocket::getBoxComments

  QHBoxLayout* DlgRocket::getBoxCtrl() {
    if (!fBoxCtrl) {
      fBoxCtrl = new QHBoxLayout();
      fBoxCtrl -> addWidget(getBtApply());
      fBoxCtrl -> addWidget(getBtCancel());
      fBoxCtrl -> addWidget(getBtReload());
      fBoxCtrl -> addWidget(getBtSave());
      fBoxCtrl -> addStretch();
      fBoxCtrl -> addWidget(getBtAbort());
      fBoxCtrl -> addWidget(getBtOK());
    } // if
    return fBoxCtrl;
  } // DlgRocket::getBoxCtrl

  QPushButton* DlgRocket::getBtAbort() {
    if (!fBtAbort) {
      fBtAbort = new QPushButton(tr("Abort"));
      fBtAbort -> setShortcut(tr("Shift+Esc"));
      connect(fBtAbort, SIGNAL(clicked()), this, SLOT(doAbort()));
    } // if
    return fBtAbort;
  } // DlgRocket::getBtAbort

  QPushButton* DlgRocket::getBtApply() {
    if (!fBtApply) {
      fBtApply = new QPushButton(tr("Apply"));
      connect(fBtApply, SIGNAL(clicked()), this, SLOT(doApply()));
    } // if
    return fBtApply;
  } // DlgRocket::getBtApply

  QPushButton* DlgRocket::getBtCancel() {
    if (!fBtCancel) {
      fBtCancel = new QPushButton(tr("Cancel input"));
      fBtCancel -> setShortcut(tr("Esc"));
      connect(fBtCancel, SIGNAL(clicked()), this, SLOT(doRefresh()));
    } // if
    return fBtCancel;
  } // DlgRocket::getBtCancel

  QPushButton* DlgRocket::getBtOK() {
    if (!fBtOK) {
      fBtOK = new QPushButton(tr("OK"));
      fBtOK -> setShortcut(tr("Shift+Return"));
      connect(fBtOK, SIGNAL(clicked()), this, SLOT(doOK()));
    } // if
    return fBtOK;
  } // DlgRocket::getBtOK

  QPushButton* DlgRocket::getBtReload() {
    if (!fBtReload) {
      fBtReload = new QPushButton(tr("Reload"));
      fBtReload -> setShortcut(tr("Ctrl+L"));
      connect(fBtReload, SIGNAL(clicked()), this, SLOT(doReload()));
    } // if
    return fBtReload;
  } // DlgRocket::getBtReload

  QPushButton* DlgRocket::getBtSave() {
    if (!fBtSave) {
      fBtSave = new QPushButton(tr("Save"));
      fBtSave -> setShortcut(tr("Ctrl+S"));
      connect(fBtSave, SIGNAL(clicked()), this, SLOT(doSave()));
    } // if
    return fBtSave;
  } // DlgRocket::getBtSave

  QComboBox* DlgRocket::getCbIce() {
    if (!fCbIce) {
      fCbIce = new QComboBox();
      connect(fCbIce, SIGNAL(currentIndexChanged(int)), this,
        SLOT(doIceChanged()));
    } // if
    return fCbIce;
  } // DlgRocket::getCbIce

  QComboBox* DlgRocket::getCbReaction() {
    if (!fCbReaction) {
      fCbReaction = new QComboBox();
      connect(fCbReaction, SIGNAL(currentIndexChanged(int)), this,
        SLOT(doReactionChanged()));
    } // if
    return fCbReaction;
  } // DlgRocket::getCbReaction

  QGridLayout* DlgRocket::getGridData() {
    if (!fGridData) {
      fGridData = new QGridLayout();
      /* Top block: */
      fGridData -> addWidget(getLbName(), 0, 0);
      fGridData -> addWidget(getLnName(), 0, 1, 1, 9);
      fGridData -> addWidget(getLbReaction(), 1, 3);
      fGridData -> addWidget(getCbReaction(), 1, 4, 1, 3);
      fGridData -> addWidget(getLbOutput(), 1, 7, 1, 2);
      fGridData -> addWidget(getLnOutput(), 1, 9);
      fGridData -> addWidget(getLbIce(), 2, 3);
      fGridData -> addWidget(getCbIce(), 2, 4, 1, 3);
      fGridData -> addWidget(getLbShareReactive(), 2, 7, 1, 2);
      fGridData -> addWidget(getLnShareReactive(), 2, 9);
      /* Main input column: */
      fGridData -> addWidget(getLbAddPropPerFuel(), 3, 3);
      fGridData -> addWidget(getLnAddPropPerFuel(), 3, 4);
      fGridData -> addWidget(getLbEfficiency(), 4, 3);
      fGridData -> addWidget(getLnEfficiency(), 4, 4);
      fGridData -> addWidget(getLbConsumption(), 5, 3);
      fGridData -> addWidget(getLnConsumption(), 5, 4);
      fGridData -> addWidget(getLbStructure(), 6, 3);
      fGridData -> addWidget(getLnStructure(), 6, 4);
      fGridData -> addWidget(getLbStartEmptyRatio(), 7, 3);
      fGridData -> addWidget(getLnStartEmptyRatio(), 7, 4);
      fGridData -> addWidget(getLbStartingMass(), 8, 3);
      fGridData -> addWidget(getLnStartingMass(), 8, 4);
      fGridData -> addWidget(getLbAccelFull(), 8, 6);
      fGridData -> addWidget(getLnAccelFull(), 8, 7);
      fGridData -> addWidget(getLbAccelEmpty(), 8, 8);
      fGridData -> addWidget(getLnAccelEmpty(), 8, 9);
      fGridData -> addWidget(getLbFuelMass(), 9, 3);
      fGridData -> addWidget(getLnFuelMass(), 9, 4);
      fGridData -> addWidget(getLbIceQuantity(), 9, 5, 1, 2);
      fGridData -> addWidget(getLnIceQuantity(), 9, 7);
      /* Thrust and power column: */
      fGridData -> addWidget(getLbPower(), 6, 0);
      fGridData -> addWidget(getLnPower(), 6, 1);
      fGridData -> addWidget(getLbDissipation(), 7, 0);
      fGridData -> addWidget(getLnDissipation(), 7, 1);
      fGridData -> addWidget(getLbThrust(), 9, 0);
      fGridData -> addWidget(getLnThrust(), 9, 1);
      /* Velocity column: */
      fGridData -> addWidget(getLbShare(), 3, 7, 1, 2);
      fGridData -> addWidget(getLnShare(), 3, 9);
      fGridData -> addWidget(getLbExhVel_c(), 5, 6);
      fGridData -> addWidget(getLnExhVel_c(), 5, 7);
      fGridData -> addWidget(getLbExhVel_km_s(), 5, 8);
      fGridData -> addWidget(getLnExhVel_km_s(), 5, 9);
      fGridData -> addWidget(getLbTrvSpd_c(), 7, 6);
      fGridData -> addWidget(getLnTrvSpd_c(), 7, 7);
      fGridData -> addWidget(getLbTrvSpd_km_s(), 7, 8);
      fGridData -> addWidget(getLnTrvSpd_km_s(), 7, 9);
      /* Recalc rules: */
      fGridData -> addWidget(getRdRecalc_w_p(), 3, 5);
      fGridData -> addWidget(getRdRecalc_w_eta(), 4, 5);
      fGridData -> addWidget(getRdRecalc_v_w(), 6, 7);
      fGridData -> addWidget(getRdRecalc_v_R(), 7, 5);
      fGridData -> addWidget(getRdRecalc_P_p(), 3, 2);
      fGridData -> addWidget(getRdRecalc_P_eta(), 4, 2);
      fGridData -> addWidget(getRdRecalc_P_mu(), 5, 2);
      /* Mission data block: */
      fGridData -> addWidget(getLbInertial(), 10, 4);
      fGridData -> addWidget(getLbOnBoard(), 10, 5);
      fGridData -> addWidget(getLbTotalFiringTime(), 11, 2, 1, 2);
      fGridData -> addWidget(getLnTotalFiringTimeInertial(), 11, 4);
      fGridData -> addWidget(getLnTotalFiringTimeOnBoard(), 11, 5);
      fGridData -> addWidget(getLbTotalFiringDistance_ly(), 11, 6);
      fGridData -> addWidget(getLnTotalFiringDistance_ly(), 11, 7);
      fGridData -> addWidget(getLbTotalFiringDistance_km(), 11, 8);
      fGridData -> addWidget(getLnTotalFiringDistance_km(), 11, 9);
      fGridData -> addWidget(getLbAccelTime(), 12, 2, 1, 2);
      fGridData -> addWidget(getLnAccelTimeInertial(), 12, 4);
      fGridData -> addWidget(getLnAccelTimeOnBoard(), 12, 5);
      fGridData -> addWidget(getLbAccelDistance_ly(), 12, 6);
      fGridData -> addWidget(getLnAccelDistance_ly(), 12, 7);
      fGridData -> addWidget(getLbAccelDistance_km(), 12, 8);
      fGridData -> addWidget(getLnAccelDistance_km(), 12, 9);
      fGridData -> addWidget(getLbDecelTime(), 13, 2, 1, 2);
      fGridData -> addWidget(getLnDecelTimeInertial(), 13, 4);
      fGridData -> addWidget(getLnDecelTimeOnBoard(), 13, 5);
      fGridData -> addWidget(getLbDecelDistance_ly(), 13, 6);
      fGridData -> addWidget(getLnDecelDistance_ly(), 13, 7);
      fGridData -> addWidget(getLbDecelDistance_km(), 13, 8);
      fGridData -> addWidget(getLnDecelDistance_km(), 13, 9);
    } // if
    return fGridData;
  } // DlgRocket::getGridData

  QButtonGroup* DlgRocket::getGrpRecalcExhVel() {
    if (!fGrpRecalcExhVel) {
      fGrpRecalcExhVel = new QButtonGroup(this);
      fGrpRecalcExhVel -> setExclusive(true);
    } // if
    return fGrpRecalcExhVel;
  } // DlgRocket::getGrpRecalcExhVel

  QButtonGroup* DlgRocket::getGrpRecalcPower() {
    if (!fGrpRecalcPower) {
      fGrpRecalcPower = new QButtonGroup(this);
      fGrpRecalcPower -> setExclusive(true);
    } // if
    return fGrpRecalcPower;
  } // DlgRocket::getGrpRecalcPower

  QButtonGroup* DlgRocket::getGrpRecalcTrvSpd() {
    if (!fGrpRecalcTrvSpd) {
      fGrpRecalcTrvSpd = new QButtonGroup(this);
      fGrpRecalcTrvSpd -> setExclusive(true);
    } // if
    return fGrpRecalcTrvSpd;
  } // DlgRocket::getGrpRecalcTrvSpd

  QLabel* DlgRocket::getLbAccelDistance_km() {
    if (!fLbAccelDistance_km) {
      fLbAccelDistance_km = new QLabel("= [km]");
      fLbAccelDistance_km -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbAccelDistance_km;
  } // DlgRocket::getLbAccelDistance_km

  QLabel* DlgRocket::getLbAccelDistance_ly() {
    if (!fLbAccelDistance_ly) {
      fLbAccelDistance_ly = new QLabel("- on a&cceleration [ly]:");
      fLbAccelDistance_ly -> setAlignment(Qt::AlignRight);
      fLbAccelDistance_ly -> setBuddy(getLnAccelDistance_ly());
    } // if
    return fLbAccelDistance_ly;
  } // DlgRocket::getLbAccelDistance_ly

  QLabel* DlgRocket::getLbAccelEmpty() {
    if (!fLbAccelEmpty) {
      fLbAccelEmpty = new QLabel(tr("to"));
      fLbAccelEmpty -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbAccelEmpty;
  } // DlgRocket::getLbAccelEmpty

  QLabel* DlgRocket::getLbAccelFull() {
    if (!fLbAccelFull) {
      fLbAccelFull = new QLabel(tr("&Acceleration [g]:"));
      fLbAccelFull -> setAlignment(Qt::AlignRight);
      fLbAccelFull -> setBuddy(getLnAccelFull());
    } // if
    return fLbAccelFull;
  } // DlgRocket:.getLbAccelFull

  QLabel* DlgRocket::getLbAccelTime() {
    if (!fLbAccelTime) {
      fLbAccelTime = new QLabel(tr("- during acceleration:"));
      fLbAccelTime -> setAlignment(Qt::AlignRight);
    } // if
    return fLbAccelTime;
  } // DlgRocket::getLbAccelTime

  QLabel* DlgRocket::getLbAddPropPerFuel() {
    if (!fLbAddPropPerFuel) {
      fLbAddPropPerFuel = new QLabel(tr("Added prope&llant per fuel:"));
      fLbAddPropPerFuel -> setAlignment(Qt::AlignRight);
      fLbAddPropPerFuel -> setBuddy(getLnAddPropPerFuel());
    } // if
    return fLbAddPropPerFuel;
  } // DlgRocket::getLbAddPropPerFuel

  QLabel* DlgRocket::getLbComments() {
    if (!fLbComments) {
      fLbComments = new QLabel(tr("Co&mments:"));
      fLbComments -> setBuddy(getTextComments());
    } // if
    return fLbComments;
  } // DlgRocket::getLbComments

  QLabel* DlgRocket::getLbConsumption() {
    if (!fLbConsumption) {
      fLbConsumption = new QLabel(tr("Consum&ption [kg/s]:"));
      fLbConsumption -> setAlignment(Qt::AlignRight);
      fLbConsumption -> setBuddy(getLnConsumption());
    } // if
    return fLbConsumption;
  } // DlgRocket::getLbConsumption

  QLabel* DlgRocket::getLbDecelDistance_km() {
    if (!fLbDecelDistance_km) {
      fLbDecelDistance_km = new QLabel("= [km]");
      fLbDecelDistance_km -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbDecelDistance_km;
  } // DlgRocket::getLbDecelDistance_km

  QLabel* DlgRocket::getLbDecelDistance_ly() {
    if (!fLbDecelDistance_ly) {
      fLbDecelDistance_ly = new QLabel("- on decelera&tion [ly]:");
      fLbDecelDistance_ly -> setAlignment(Qt::AlignRight);
      fLbDecelDistance_ly -> setBuddy(getLnDecelDistance_ly());
    } // if
    return fLbDecelDistance_ly;
  } // DlgRocket::getLbDecelDistance_ly

  QLabel* DlgRocket::getLbDecelTime() {
    if (!fLbDecelTime) {
      fLbDecelTime = new QLabel(tr("- during deceleration:"));
      fLbDecelTime -> setAlignment(Qt::AlignRight);
    } // if
    return fLbDecelTime;
  } // DlgRocket::getLbDecelTime

  QLabel* DlgRocket::getLbDissipation() {
    if (!fLbDissipation) {
      fLbDissipation = new QLabel(tr("&Dissipation [W]:"));
      fLbDissipation -> setAlignment(Qt::AlignRight);
      fLbDissipation -> setBuddy(getLnDissipation());
    } // if
    return fLbDissipation;
  } // DlgRocket::getLbDissipation

  QLabel* DlgRocket::getLbEfficiency() {
    if (!fLbEfficiency) {
      fLbEfficiency = new QLabel(tr("Efficienc&y:"));
      fLbEfficiency -> setAlignment(Qt::AlignRight);
      fLbEfficiency -> setBuddy(getLnEfficiency());
    } // if
    return fLbEfficiency;
  } // DlgRocket::getLbEfficiency

  QLabel* DlgRocket::getLbExhVel_c() {
    if (!fLbExhVel_c) {
      fLbExhVel_c = new QLabel(tr("E&xhaust velocity [c]:"));
      fLbExhVel_c -> setAlignment(Qt::AlignRight);
      fLbExhVel_c -> setBuddy(getLnExhVel_c());
    } // if
    return fLbExhVel_c;
  } // DlgRocket::getLbExhVel_c

  QLabel* DlgRocket::getLbExhVel_km_s() {
    if (!fLbExhVel_km_s) {
      fLbExhVel_km_s = new QLabel(tr("= [km/s]"));
      fLbExhVel_km_s -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbExhVel_km_s;
  } // DlgRocket::getLbExhVel_km_s

  QLabel* DlgRocket::getLbFuelMass() {
    if (!fLbFuelMass) {
      fLbFuelMass = new QLabel(tr("Fu&el mass [t]:"));
      fLbFuelMass -> setAlignment(Qt::AlignRight);
      fLbFuelMass -> setBuddy(getLnFuelMass());
    } // if
    return fLbFuelMass;
  } // DlgRocket::getLbFuelMass

  QLabel* DlgRocket::getLbIce() {
    if (!fLbIce) {
      fLbIce = new QLabel(tr("&Ice type:"));
      fLbIce -> setAlignment(Qt::AlignRight);
      fLbIce -> setBuddy(getCbIce());
    } // if
    return fLbIce;
  } // DlgRocket::getLbIce

  QLabel* DlgRocket::getLbIceQuantity() {
    if (!fLbIceQuantity) {
      fLbIceQuantity = new QLabel(tr("Ice &quantity diameter [km]:"));
      fLbIceQuantity -> setAlignment(Qt::AlignRight);
      fLbIceQuantity -> setBuddy(getLnIceQuantity());
    } // if
    return fLbIceQuantity;
  } // DlgRocket::getLbIceQuantity

  QLabel* DlgRocket::getLbInertial() {
    if (!fLbInertial) {
      fLbInertial = new QLabel(tr("inertial"));
      fLbInertial -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbInertial;
  } // DlgRocket::getLbInertial

  QLabel* DlgRocket::getLbName() {
    if (!fLbName) {
      fLbName = new QLabel(tr("&Name:"));
      fLbName -> setAlignment(Qt::AlignRight);
      fLbName -> setBuddy(getLnName());
    } // if
    return fLbName;
  } // DlgRocket::getLbName

  QLabel* DlgRocket::getLbOnBoard() {
    if (!fLbOnBoard) {
      fLbOnBoard = new QLabel(tr("on-board"));
      fLbOnBoard -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbOnBoard;
  } // DlgRocket::getLbOnBoard

  QLabel* DlgRocket::getLbOutput() {
    if (!fLbOutput) {
      fLbOutput = new QLabel(tr("Output:"));
      fLbOutput -> setAlignment(Qt::AlignRight);
    } // if
    return fLbOutput;
  } // DlgRocket::getLbOutput

  QLabel* DlgRocket::getLbPower() {
    if (!fLbPower) {
      fLbPower = new QLabel(tr("Po&wer [W]:"));
      fLbPower -> setAlignment(Qt::AlignRight);
      fLbPower -> setBuddy(getLnPower());
    } // if
    return fLbPower;
  } // DlgRocket::getLbPower

  QLabel* DlgRocket::getLbReaction() {
    if (!fLbReaction) {
      fLbReaction = new QLabel(tr("&Reaction:"));
      fLbReaction -> setAlignment(Qt::AlignRight);
      fLbReaction -> setBuddy(getCbReaction());
    } // if
    return fLbReaction;
  } // DlgRocket::getLbReaction

  QLabel* DlgRocket::getLbShare() {
    if (!fLbShare) {
      fLbShare = new QLabel(tr("Total reactive &share:"));
      fLbShare -> setAlignment(Qt::AlignRight);
      fLbShare -> setBuddy(getLnShare());
    } // if
    return fLbShare;
  } // DlgRocket::getLbShare

  QLabel* DlgRocket::getLbShareReactive() {
    if (!fLbShareReactive) {
      fLbShareReactive = new QLabel(tr("Ice reactive share:"));
      fLbShareReactive -> setAlignment(Qt::AlignRight);
    } // if
    return fLbShareReactive;
  } // DlgRocket::getLbShareReactive

  QLabel* DlgRocket::getLbStartEmptyRatio() {
    if (!fLbStartEmptyRatio) {
      fLbStartEmptyRatio = new QLabel(tr("Start&/Empty ratio:"));
      fLbStartEmptyRatio -> setAlignment(Qt::AlignRight);
      fLbStartEmptyRatio -> setBuddy(getLnStartEmptyRatio());
    } // if
    return fLbStartEmptyRatio;
  } // DlgRocket::getLbStartEmptyRatio

  QLabel* DlgRocket::getLbStartingMass() {
    if (!fLbStartingMass) {
      fLbStartingMass = new QLabel(tr("Startin&g mass [t]:"));
      fLbStartingMass -> setAlignment(Qt::AlignRight);
      fLbStartingMass -> setBuddy(getLnStartingMass());
    } // if
    return fLbStartingMass;
  } // DlgRocket::getLbStartingMass

  QLabel* DlgRocket::getLbStructure() {
    if (!fLbStructure) {
      fLbStructure = new QLabel(tr("Str&ucture mass [t]:"));
      fLbStructure -> setAlignment(Qt::AlignRight);
      fLbStructure -> setBuddy(getLnStructure());
    } // if
    return fLbStructure;
  } // DlgRocket::getLbStructure

  QLabel* DlgRocket::getLbThrust() {
    if (!fLbThrust) {
      fLbThrust = new QLabel(tr("T&hrust [N]:"));
      fLbThrust -> setAlignment(Qt::AlignRight);
      fLbThrust -> setBuddy(getLnThrust());
    } // if
    return fLbThrust;
  } // DlgRocket::getLbThrust

  QLabel* DlgRocket::getLbTotalFiringDistance_km() {
    if (!fLbTotalFiringDistance_km) {
      fLbTotalFiringDistance_km = new QLabel("= [km]");
      fLbTotalFiringDistance_km -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbTotalFiringDistance_km;
  } // DlgRocket::getLbTotalFiringDistance_km

  QLabel* DlgRocket::getLbTotalFiringDistance_ly() {
    if (!fLbTotalFiringDistance_ly) {
      fLbTotalFiringDistance_ly = new QLabel("Distance c&overed [ly]:");
      fLbTotalFiringDistance_ly -> setAlignment(Qt::AlignRight);
      fLbTotalFiringDistance_ly -> setBuddy(getLnTotalFiringDistance_ly());
    } // if
    return fLbTotalFiringDistance_ly;
  } // DlgRocket::getLbTotalFiringDistance_ly

  QLabel* DlgRocket::getLbTotalFiringTime() {
    if (!fLbTotalFiringTime) {
      fLbTotalFiringTime = new QLabel(tr("Total firing time of drives:"));
      fLbTotalFiringTime -> setAlignment(Qt::AlignRight);
    } // if
    return fLbTotalFiringTime;
  } // DlgRocket::getLbTotalFiringTime

  QLabel* DlgRocket::getLbTrvSpd_c() {
    if (!fLbTrvSpd_c) {
      fLbTrvSpd_c = new QLabel(tr("Tra&vel speed [c]:"));
      fLbTrvSpd_c -> setAlignment(Qt::AlignRight);
      fLbTrvSpd_c -> setBuddy(getLnTrvSpd_c());
    } // if
    return fLbTrvSpd_c;
  } // DlgRocket::getLbTrvSpd_c

  QLabel* DlgRocket::getLbTrvSpd_km_s() {
    if (!fLbTrvSpd_km_s) {
      fLbTrvSpd_km_s = new QLabel(tr("= [km/s]"));
      fLbTrvSpd_km_s -> setAlignment(Qt::AlignCenter);
    } // if
    return fLbTrvSpd_km_s;
  } // DlgRocket::getLbTrvSpd_km_s

  QLineEdit *DlgRocket::getLnAccelDistance_ly() {
    if (!fLnAccelDistance_ly) {
      fLnAccelDistance_ly = new QLineEdit();
      fLnAccelDistance_ly -> setMinimumWidth(EDIT_WIDTH);
      fLnAccelDistance_ly -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnAccelDistance_ly, SIGNAL(editingFinished()), this,
        SLOT(doFetchAccelDistance_ly()));
    } // if
    return fLnAccelDistance_ly;
  } // DlgRocket::getLnAccelDistance_ly

  QLineEdit *DlgRocket::getLnAccelDistance_km() {
    if (!fLnAccelDistance_km) {
      fLnAccelDistance_km = new QLineEdit();
      fLnAccelDistance_km -> setMinimumWidth(EDIT_WIDTH);
      fLnAccelDistance_km -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnAccelDistance_km, SIGNAL(editingFinished()), this,
        SLOT(doFetchAccelDistance_km()));
    } // if
    return fLnAccelDistance_km;
  } // DlgRocket::getLnAccelDistance_km

  QLineEdit* DlgRocket::getLnAccelEmpty() {
    if (!fLnAccelEmpty) {
      fLnAccelEmpty = new QLineEdit();
      fLnAccelEmpty -> setMinimumWidth(EDIT_WIDTH);
      fLnAccelEmpty -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnAccelEmpty, SIGNAL(editingFinished()), this,
        SLOT(doFetchAccelEmpty()));
    } // if
    return fLnAccelEmpty;
  } // DlgRocket::getLnAccelEmpty

  QLineEdit* DlgRocket::getLnAccelFull() {
    if (!fLnAccelFull) {
      fLnAccelFull = new QLineEdit();
      fLnAccelFull -> setMinimumWidth(EDIT_WIDTH);
      fLnAccelFull -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnAccelFull, SIGNAL(editingFinished()), this,
        SLOT(doFetchAccelFull()));
    } // if
    return fLnAccelFull;
  } // DlgRocket::getLnAccelFull

  QLineEdit *DlgRocket::getLnAccelTimeInertial() {
    if (!fLnAccelTimeInertial) {
      fLnAccelTimeInertial = new QLineEdit();
      fLnAccelTimeInertial -> setMinimumWidth(EDIT_WIDTH);
      fLnAccelTimeInertial -> setMaximumWidth(EDIT_WIDTH);
      fLnAccelTimeInertial -> setReadOnly(true);
    } // if
    return fLnAccelTimeInertial;
  } // DlgRocket::getLnAccelTimeInertial

  QLineEdit *DlgRocket::getLnAccelTimeOnBoard() {
    if (!fLnAccelTimeOnBoard) {
      fLnAccelTimeOnBoard = new QLineEdit();
      fLnAccelTimeOnBoard -> setMinimumWidth(EDIT_WIDTH);
      fLnAccelTimeOnBoard -> setMaximumWidth(EDIT_WIDTH);
      fLnAccelTimeOnBoard -> setReadOnly(true);
    } // if
    return fLnAccelTimeOnBoard;
  } // DlgRocket::getLnAccelTimeOnBoard

  QLineEdit* DlgRocket::getLnAddPropPerFuel() {
    if (!fLnAddPropPerFuel) {
      fLnAddPropPerFuel = new QLineEdit();
      fLnAddPropPerFuel -> setMinimumWidth(EDIT_WIDTH);
      fLnAddPropPerFuel -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnAddPropPerFuel, SIGNAL(editingFinished()), this,
        SLOT(doFetchAddPropPerFuel()));
    } // if
    return fLnAddPropPerFuel;
  } // DlgRocket::getLnAddPropPerFuel

  QLineEdit* DlgRocket::getLnConsumption() {
    if (!fLnConsumption) {
      fLnConsumption = new QLineEdit();
      fLnConsumption -> setMinimumWidth(EDIT_WIDTH);
      fLnConsumption -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnConsumption, SIGNAL(editingFinished()), this,
        SLOT(doFetchConsumption()));
    } // if
    return fLnConsumption;
  } // DlgRocket::getLnConsumption

  QLineEdit *DlgRocket::getLnDecelDistance_ly() {
    if (!fLnDecelDistance_ly) {
      fLnDecelDistance_ly = new QLineEdit();
      fLnDecelDistance_ly -> setMinimumWidth(EDIT_WIDTH);
      fLnDecelDistance_ly -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnDecelDistance_ly, SIGNAL(editingFinished()), this,
        SLOT(doFetchDecelDistance_ly()));
    } // if
    return fLnDecelDistance_ly;
  } // DlgRocket::getLnDecelDistance_ly

  QLineEdit *DlgRocket::getLnDecelDistance_km() {
    if (!fLnDecelDistance_km) {
      fLnDecelDistance_km = new QLineEdit();
      fLnDecelDistance_km -> setMinimumWidth(EDIT_WIDTH);
      fLnDecelDistance_km -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnDecelDistance_km, SIGNAL(editingFinished()), this,
        SLOT(doFetchDecelDistance_km()));
    } // if
    return fLnDecelDistance_km;
  } // DlgRocket::getLnDecelDistance_km

  QLineEdit *DlgRocket::getLnDecelTimeInertial() {
    if (!fLnDecelTimeInertial) {
      fLnDecelTimeInertial = new QLineEdit();
      fLnDecelTimeInertial -> setMinimumWidth(EDIT_WIDTH);
      fLnDecelTimeInertial -> setMaximumWidth(EDIT_WIDTH);
      fLnDecelTimeInertial -> setReadOnly(true);
    } // if
    return fLnDecelTimeInertial;
  } // DlgRocket::getLnDecelTimeInertial

  QLineEdit *DlgRocket::getLnDecelTimeOnBoard() {
    if (!fLnDecelTimeOnBoard) {
      fLnDecelTimeOnBoard = new QLineEdit();
      fLnDecelTimeOnBoard -> setMinimumWidth(EDIT_WIDTH);
      fLnDecelTimeOnBoard -> setMaximumWidth(EDIT_WIDTH);
      fLnDecelTimeOnBoard -> setReadOnly(true);
    } // if
    return fLnDecelTimeOnBoard;
  } // DlgRocket::getLnDecelTimeOnBoard

  QLineEdit* DlgRocket::getLnDissipation() {
    if (!fLnDissipation) {
      fLnDissipation = new QLineEdit();
      fLnDissipation -> setMinimumWidth(EDIT_WIDTH);
      fLnDissipation -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnDissipation, SIGNAL(editingFinished()), this,
        SLOT(doFetchDissipation()));
    } // if
    return fLnDissipation;
  } // DlgRocket::getLnDissipation

  QLineEdit* DlgRocket::getLnEfficiency() {
    if (!fLnEfficiency) {
      fLnEfficiency = new QLineEdit();
      fLnEfficiency -> setMinimumWidth(EDIT_WIDTH);
      fLnEfficiency -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnEfficiency, SIGNAL(editingFinished()), this,
        SLOT(doFetchEfficiency()));
    } // if
    return fLnEfficiency;
  } // DlgRocket::getLnEfficiency

  QLineEdit* DlgRocket::getLnExhVel_c() {
    if (!fLnExhVel_c) {
      fLnExhVel_c = new QLineEdit();
      fLnExhVel_c -> setMinimumWidth(EDIT_WIDTH);
      fLnExhVel_c -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnExhVel_c, SIGNAL(editingFinished()), this, SLOT(doFetchExhVel_c()));
    } // if
    return fLnExhVel_c;
  } // DlgRocket::getLnExhVel_c

  QLineEdit* DlgRocket::getLnExhVel_km_s() {
    if (!fLnExhVel_km_s) {
      fLnExhVel_km_s = new QLineEdit();
      fLnExhVel_km_s -> setMinimumWidth(EDIT_WIDTH);
      fLnExhVel_km_s -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnExhVel_km_s, SIGNAL(editingFinished()), this,
        SLOT(doFetchExhVel_km_s()));
    } // if
    return fLnExhVel_km_s;
  } // DlgRocket::getLnExhVel_km_s

  QLineEdit* DlgRocket::getLnFuelMass() {
    if (!fLnFuelMass) {
      fLnFuelMass = new QLineEdit();
      fLnFuelMass -> setMinimumWidth(EDIT_WIDTH);
      fLnFuelMass -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnFuelMass, SIGNAL(editingFinished()), this,
        SLOT(doFetchFuelMass()));
    } // if
    return fLnFuelMass;
  } // DlgRocket::getLnFuelMass

  QLineEdit* DlgRocket::getLnIceQuantity() {
    if (!fLnIceQuantity) {
      fLnIceQuantity = new QLineEdit();
      fLnIceQuantity -> setMinimumWidth(EDIT_WIDTH);
      fLnIceQuantity -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnIceQuantity, SIGNAL(editingFinished()), this,
        SLOT(doFetchIceQuantity()));
    } // if
    return fLnIceQuantity;
  } // DlgRocket::getLnIceQuantity

  QLineEdit* DlgRocket::getLnName() {
    if (!fLnName) {
      fLnName = new QLineEdit();
      connect(fLnName, SIGNAL(editingFinished()), this, SLOT(doFetchName()));
    } // if
    return fLnName;
  } // DlgRocket::getLnName

  QLineEdit* DlgRocket::getLnOutput() {
    if (!fLnOutput) {
      fLnOutput = new QLineEdit();
      fLnOutput -> setMinimumWidth(EDIT_WIDTH);
      fLnOutput -> setMaximumWidth(EDIT_WIDTH);
      fLnOutput -> setReadOnly(true);
    } // if
    return fLnOutput;
  } // DlgRocket::getLnOutput

  QLineEdit* DlgRocket::getLnPower() {
    if (!fLnPower) {
      fLnPower = new QLineEdit();
      fLnPower -> setMinimumWidth(EDIT_WIDTH);
      fLnPower -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnPower, SIGNAL(editingFinished()), this, SLOT(doFetchPower()));
    } // if
    return fLnPower;
  } // DlgRocket::getLnPower

  QLineEdit* DlgRocket::getLnShare() {
    if (!fLnShare) {
      fLnShare = new QLineEdit();
      fLnShare -> setMinimumWidth(EDIT_WIDTH);
      fLnShare -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnShare, SIGNAL(editingFinished()), this,
        SLOT(doFetchReactiveShare()));
    } // if
    return fLnShare;
  } // DlgRocket::getLnShare

  QLineEdit* DlgRocket::getLnShareReactive() {
    if (!fLnShareReactive) {
      fLnShareReactive = new QLineEdit();
      fLnShareReactive -> setMinimumWidth(EDIT_WIDTH);
      fLnShareReactive -> setMaximumWidth(EDIT_WIDTH);
      fLnShareReactive -> setReadOnly(true);
    } // if
    return fLnShareReactive;
  } // DlgRocket::getLnShareReactive

  QLineEdit* DlgRocket::getLnStartEmptyRatio() {
    if (!fLnStartEmptyRatio) {
      fLnStartEmptyRatio = new QLineEdit();
      fLnStartEmptyRatio -> setMinimumWidth(EDIT_WIDTH);
      fLnStartEmptyRatio -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnStartEmptyRatio, SIGNAL(editingFinished()), this,
        SLOT(doFetchStartEmptyRatio()));
    } // if
    return fLnStartEmptyRatio;
  } // DlgRocket::getLnStartEmptyRatio

  QLineEdit* DlgRocket::getLnStartingMass() {
    if (!fLnStartingMass) {
      fLnStartingMass = new QLineEdit();
      fLnStartingMass -> setMinimumWidth(EDIT_WIDTH);
      fLnStartingMass -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnStartingMass, SIGNAL(editingFinished()), this,
        SLOT(doFetchStartingMass()));
    } // if
    return fLnStartingMass;
  } // DlgRocket::getLnStartingMass

  QLineEdit* DlgRocket::getLnStructure() {
    if (!fLnStructure) {
      fLnStructure = new QLineEdit();
      fLnStructure -> setMinimumWidth(EDIT_WIDTH);
      fLnStructure -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnStructure, SIGNAL(editingFinished()), this,
        SLOT(doFetchStructure()));
    } // if
    return fLnStructure;
  } // DlgRocket::getLnStructure

  QLineEdit* DlgRocket::getLnThrust() {
    if (!fLnThrust) {
      fLnThrust = new QLineEdit();
      fLnThrust -> setMinimumWidth(EDIT_WIDTH);
      fLnThrust -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnThrust, SIGNAL(editingFinished()), this, SLOT(doFetchThrust()));
    } // if
    return fLnThrust;
  } // DlgRocket::getLnThrust

  QLineEdit *DlgRocket::getLnTotalFiringDistance_ly() {
    if (!fLnTotalFiringDistance_ly) {
      fLnTotalFiringDistance_ly = new QLineEdit();
      fLnTotalFiringDistance_ly -> setMinimumWidth(EDIT_WIDTH);
      fLnTotalFiringDistance_ly -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnTotalFiringDistance_ly, SIGNAL(editingFinished()), this,
        SLOT(doFetchTotalFiringDistance_ly()));
    } // if
    return fLnTotalFiringDistance_ly;
  } // DlgRocket::getLnTotalFiringDistance_ly

  QLineEdit *DlgRocket::getLnTotalFiringDistance_km() {
    if (!fLnTotalFiringDistance_km) {
      fLnTotalFiringDistance_km = new QLineEdit();
      fLnTotalFiringDistance_km -> setMinimumWidth(EDIT_WIDTH);
      fLnTotalFiringDistance_km -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnTotalFiringDistance_km, SIGNAL(editingFinished()), this,
        SLOT(doFetchTotalFiringDistance_km()));
    } // if
    return fLnTotalFiringDistance_km;
  } // DlgRocket::getLnTotalFiringDistance_km

  QLineEdit *DlgRocket::getLnTotalFiringTimeInertial() {
    if (!fLnTotalFiringTimeInertial) {
      fLnTotalFiringTimeInertial = new QLineEdit();
      fLnTotalFiringTimeInertial -> setMinimumWidth(EDIT_WIDTH);
      fLnTotalFiringTimeInertial -> setMaximumWidth(EDIT_WIDTH);
      fLnTotalFiringTimeInertial -> setReadOnly(true);
    } // if
    return fLnTotalFiringTimeInertial;
  } // DlgRocket::getLnTotalFiringTimeInertial

  QLineEdit *DlgRocket::getLnTotalFiringTimeOnBoard() {
    if (!fLnTotalFiringTimeOnBoard) {
      fLnTotalFiringTimeOnBoard = new QLineEdit();
      fLnTotalFiringTimeOnBoard -> setMinimumWidth(EDIT_WIDTH);
      fLnTotalFiringTimeOnBoard -> setMaximumWidth(EDIT_WIDTH);
      fLnTotalFiringTimeOnBoard -> setReadOnly(true);
    } // if
    return fLnTotalFiringTimeOnBoard;
  } // DlgRocket::getLnTotalFiringTimeOnBoard

  QLineEdit* DlgRocket::getLnTrvSpd_c() {
    if (!fLnTrvSpd_c) {
      fLnTrvSpd_c = new QLineEdit();
      fLnTrvSpd_c -> setMinimumWidth(EDIT_WIDTH);
      fLnTrvSpd_c -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnTrvSpd_c, SIGNAL(editingFinished()), this, SLOT(doFetchTrvSpd_c()));
    } // if
    return fLnTrvSpd_c;
  } // DlgRocket::getLnTrvSpd_c

  QLineEdit* DlgRocket::getLnTrvSpd_km_s() {
    if (!fLnTrvSpd_km_s) {
      fLnTrvSpd_km_s = new QLineEdit();
      fLnTrvSpd_km_s -> setMinimumWidth(EDIT_WIDTH);
      fLnTrvSpd_km_s -> setMaximumWidth(EDIT_WIDTH);
      connect(fLnTrvSpd_km_s, SIGNAL(editingFinished()), this,
        SLOT(doFetchTrvSpd_km_s()));
    } // if
    return fLnTrvSpd_km_s;
  } // DlgRocket::getLnTrvSpd_km_s

  RecalcPath* DlgRocket::getPath_P_eta() {
    if (!fPath_P_eta) {
      fPath_P_eta = new RecalcPath(getGridData(), 7, 1, 4, 4, Qt::blue,
        Qt::Vertical);
    } // if
    return fPath_P_eta;
  } // DlgRocket::getPath_P_eta

  RecalcPath* DlgRocket::getPath_P_mu() {
    if (!fPath_P_mu) {
      fPath_P_mu = new RecalcPath(getGridData(), 7, 1, 5, 4, Qt::blue,
        Qt::Vertical);
    } // if
    return fPath_P_mu;
  } // DlgRocket::getPath_P_mu

  RecalcPath* DlgRocket::getPath_P_p() {
    if (!fPath_P_p) {
      fPath_P_p = new RecalcPath(getGridData(), 7, 1, 3, 4, Qt::blue,
        Qt::Vertical);
    } // if
    return fPath_P_p;
  } // DlgRocket::getPath_P_p

  RecalcPath* DlgRocket::getPath_v_R() {
    if (!fPath_v_R) {
      fPath_v_R = new RecalcPath(getGridData(), 7, 7, 7, 4, Qt::green,
        Qt::Vertical);
    } // if
    return fPath_v_R;
  } // DlgRocket::getPath_v_R

  RecalcPath* DlgRocket::getPath_v_w() {
    if (!fPath_v_w) {
      fPath_v_w = new RecalcPath(getGridData(), 7, 7, 5, 7, Qt::green,
        Qt::Vertical);
    } // if
    return fPath_v_w;
  } // DlgRocket::getPath_v_w

  RecalcPath* DlgRocket::getPath_w_eta() {
    if (!fPath_w_eta) {
      fPath_w_eta = new RecalcPath(getGridData(), 5, 7, 4, 4, Qt::red,
        Qt::Vertical);
    } // if
    return fPath_w_eta;
  } // DlgRocket::getPath_w_eta

  RecalcPath* DlgRocket::getPath_w_p() {
    if (!fPath_w_p) {
      fPath_w_p = new RecalcPath(getGridData(), 5, 7, 3, 4, Qt::red,
        Qt::Vertical);
    } // if
    return fPath_w_p;
  } // DlgRocket::getPath_w_p

  QRadioButton* DlgRocket::getRdRecalc_P_eta() {
    if (!fRdRecalc_P_eta) {
      fRdRecalc_P_eta = new QRadioButton("&2");
      fRdRecalc_P_eta -> setFont(fFtRecalc);
      getGrpRecalcPower() -> addButton(fRdRecalc_P_eta);
      connect(fRdRecalc_P_eta, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_P_eta;
  } // DlgRocket::getRdRecalc_P_eta

  QRadioButton* DlgRocket::getRdRecalc_P_mu() {
    if (!fRdRecalc_P_mu) {
      fRdRecalc_P_mu = new QRadioButton("&3");
      fRdRecalc_P_mu -> setFont(fFtRecalc);
      getGrpRecalcPower() -> addButton(fRdRecalc_P_mu);
      connect(fRdRecalc_P_mu, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_P_mu;
  } // DlgRocket::getRdRecalc_P_mu

  QRadioButton* DlgRocket::getRdRecalc_P_p() {
    if (!fRdRecalc_P_p) {
      fRdRecalc_P_p = new QRadioButton("&1");
      fRdRecalc_P_p -> setFont(fFtRecalc);
      getGrpRecalcPower() -> addButton(fRdRecalc_P_p);
      connect(fRdRecalc_P_p, SIGNAL(toggled(bool)), this, SLOT(repaint()));
      fRdRecalc_P_p -> setChecked(true);
    } // if
    return fRdRecalc_P_p;
  } // DlgRocket::getRdRecalc_P_p

  QRadioButton* DlgRocket::getRdRecalc_v_R() {
    if (!fRdRecalc_v_R) {
      fRdRecalc_v_R = new QRadioButton("&7");
      fRdRecalc_v_R -> setFont(fFtRecalc);
      getGrpRecalcTrvSpd() -> addButton(fRdRecalc_v_R);
      connect(fRdRecalc_v_R, SIGNAL(toggled(bool)), this, SLOT(repaint()));
      fRdRecalc_v_R -> setChecked(true);
    } // if
    return fRdRecalc_v_R;
  } // DlgRocket::getRdRecalc_v_R

  QRadioButton* DlgRocket::getRdRecalc_v_w() {
    if (!fRdRecalc_v_w) {
      fRdRecalc_v_w = new QRadioButton("&6");
      fRdRecalc_v_w -> setFont(fFtRecalc);
      getGrpRecalcTrvSpd() -> addButton(fRdRecalc_v_w);
      connect(fRdRecalc_v_w, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_v_w;
  } // DlgRocket::getRdRecalc_v_w

  QRadioButton* DlgRocket::getRdRecalc_w_eta() {
    if (!fRdRecalc_w_eta) {
      fRdRecalc_w_eta = new QRadioButton("&5");
      fRdRecalc_w_eta -> setFont(fFtRecalc);
      getGrpRecalcExhVel() -> addButton(fRdRecalc_w_eta);
      connect(fRdRecalc_w_eta, SIGNAL(toggled(bool)), this, SLOT(repaint()));
    } // if
    return fRdRecalc_w_eta;
  } // DlgRocket::getRdRecalc_w_eta

  QRadioButton* DlgRocket::getRdRecalc_w_p() {
    if (!fRdRecalc_w_p) {
      fRdRecalc_w_p = new QRadioButton("&4");
      fRdRecalc_w_p -> setFont(fFtRecalc);
      getGrpRecalcExhVel() -> addButton(fRdRecalc_w_p);
      connect(fRdRecalc_w_p, SIGNAL(toggled(bool)), this, SLOT(repaint()));
      fRdRecalc_w_p -> setChecked(true);
    } // if
    return fRdRecalc_w_p;
  } // DlgRocket::getRdRecalc_w_p

  Ship::RecalcRule DlgRocket::getRecalcExhVel() {
    if (getRdRecalc_w_p() -> isChecked()) {
      return Ship::rrByShare;
    } else {
      return Ship::rrByEfficiency;
    } // if
  } // DlgRocket::getRecalcExhVel

  Ship::RecalcRule DlgRocket::getRecalcPower() {
    if (getRdRecalc_P_p() -> isChecked()) {
      return Ship::rrByShare;
    } else if (getRdRecalc_P_eta() -> isChecked()) {
      return Ship::rrByEfficiency;
    } else {
      return Ship::rrByFuel;
    } // if
  } // DlgRocket::getRecalcPower

  Ship::RecalcRule DlgRocket::getRecalcTrvSpd() {
    if (getRdRecalc_v_w() -> isChecked()) {
      return getRecalcExhVel();
    } else {
      return Ship::rrByFuel;
    } // if
  } // DlgRocket::getRecalcTrvSpd

  Rocket* DlgRocket::getRocket() {
    return fRocket;
  } // DlgRocket::getRocket

  QTextEdit* DlgRocket::getTextComments() {
    if (!fTextComments) {
      fTextComments = new QTextEdit();
/*!!!
      connect(fTextComments, SIGNAL(textChanged()), this,
        SLOT(doFetchComments()));
!!!*/
    } // if
    return fTextComments;
  } // DlgRocket::getTextComments

  void DlgRocket::paintEvent(QPaintEvent *aEvent) {
    QDialog::paintEvent(aEvent);
    QPainter painter(this);
    Ship::RecalcRule rr_w = getRecalcExhVel();
    Ship::RecalcRule rr_v = getRecalcTrvSpd();
    Ship::RecalcRule rr_P = getRecalcPower();
    getPath_w_p() -> draw(painter, rr_w == Ship::rrByShare);
    getPath_w_eta() -> draw(painter, rr_w == Ship::rrByEfficiency);
    getPath_v_w() -> draw(painter,
      rr_v == Ship::rrByShare || rr_v == Ship::rrByEfficiency);
    getPath_v_R() -> draw(painter, rr_v == Ship::rrByFuel);
    getPath_P_p() -> draw(painter, rr_P == Ship::rrByShare);
    getPath_P_eta() -> draw(painter, rr_P == Ship::rrByEfficiency);
    getPath_P_mu() -> draw(painter, rr_P == Ship::rrByFuel);
  } // DlgRocket::paintEvent

  void DlgRocket::reInitCbIce() {
    QList <Ice*> ices;
    Ice *ice;
    /* Sort ice types alphabetically: */
    GerQ::InstContainer *ctr = Ice::getTheContainer();
    GerQ::DataIterator iter = ctr -> beginData();
    while (iter != ctr -> endData()) {
      Ice *curIce = (Ice*) *iter;
      int i = 0;
      bool found = false;
      while (i < ices.count() && !found) {
        ice = ices[i];
	found = ice -> getName() > curIce -> getName();
	if (!found) {
	  i++;
	} // if
      } // while
      ices.insert(i, curIce);
      iter++;
    } // while
    /* Insert sorted ice types into combo: */
    QComboBox *combo = getCbIce();
    combo -> clear();
    combo -> addItem(tr("<none>"), QVariant());
    while (ices.count() > 0) {
      ice = *(ices.begin());
      combo -> addItem(ice -> getName(), ice -> getID());
      ices.erase(ices.begin());
    } // while
  } // DlgRocket::reInitCbIce

  void DlgRocket::reInitCbReaction() {
    GerFuSp::reInitCbReaction(getCbReaction(), tr("<none>"));
  } // DlgRocket::reInitCbReaction

  void DlgRocket::resetTitle() {
    if (fRocket) {
      setWindowTitle(QString().sprintf(qPrintable(tr(TITLE_FULL)),
        qPrintable(IDToStr(fRocket -> getID())),
	qPrintable(fRocket -> getName())));
    } else {
      setWindowTitle(tr(TITLE_EMPTY));
    } // if
  } // DlgRocket::resetTitle

  void DlgRocket::setRocket(Rocket *aValue) {
    fRocket = aValue;
    resetTitle();
    doRefresh();
    QLineEdit *edit = getLnName();
    edit -> selectAll();
    edit -> setFocus();
  } // DlgRocket::setRocket

  void DlgRocket::showException(const std::exception &aException) {
    QMessageBox::information(this, tr("Error"), tr(aException.what()));
  } // DlgRocket::showException

} // GerFuSp
