/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "gerfuspbase.h"

#include "reaction.h"
#include <gerph/si.h>
#include <gerq/instcontainer.h>
#include <QModelIndex>
#include <QString>
#include <stdexcept>


namespace GerFuSp {

  int curTableRow(QTableView *aTable) {
    return aTable -> currentIndex().row();
  } // curTableRow

  double dblFromLineEdit(QLineEdit *aEdit) {
    return DoubleOrExc(aEdit -> text());
  } // dblFromLineEdit

  double DoubleOrExc(const QString &aValue) {
    if (aValue == "") {
      return 0.0;
    } // if
    bool ok = true;
    double x = aValue.toDouble(&ok);
    if (ok) {
      return x;
    } else {
      throw std::invalid_argument(qPrintable(QString().sprintf(
	"Not a valid double: \"%s\"", qPrintable(aValue))));
    } // if
  } // DoubleOrExc

  QString DurationToStr(double aDuration) {
    double y = aDuration / GerPh::SI::unit_year;
    if (y >= 1.0e4) {
      return QString().sprintf("%0.2ey");
    } else if (y >= 1.0) {
      return QString().sprintf("%0.2fy", y);
    } // if
    double d = aDuration / GerPh::SI::unit_day;
    if (d >= 1.0) {
      return QString().sprintf("%0.2fd", d);
    } // if
    double h = aDuration / GerPh::SI::unit_h;
    if (h >= 0.1) {
      return QString().sprintf("%0.2fh", h);
    } else {
      return QString().sprintf("%0.2eh", h);
    } // if
  } // DurationToStr

  void gotoComboItem(QComboBox *aCombo, GerQ::DataObject *aData) {
    int i = 0;
    int n = aCombo -> count();
    bool found = false;
    QVariant ID = QVariant();
    if (aData) {
      ID = aData -> getID();
    } // if
    while (i < n && !found) {
      found = aCombo -> itemData(i) == ID;
      if (found) {
        aCombo -> setCurrentIndex(i);
      } else {
        i++;
      } // if
    } // while
  } // gotoComboItem

  void gotoTableRow(QTableView *aTable, int aRow) {
    int col = aTable -> currentIndex().column();
    QModelIndex idx = aTable -> model() -> index(aRow, col);
    aTable -> setCurrentIndex(idx);
  } // gotoTableRow

  QString IDToStr(const QVariant &aID) {
    return QString().sprintf("#%0.4i", aID.toInt());
  } // IDToStr

  void initLineEdit(QLineEdit *aEdit, const QString &aText, bool aReadOnly) {
    if (!aEdit) {
      return;
    } // if
    aEdit -> setText(aText);
    aEdit -> setReadOnly(aReadOnly);
  } // initLineEdit

  void initLineEditF(QLineEdit *aEdit, double aValue, const char* aFormat,
    double aMin, double aMax)
  {
    QString text;
    if (aValue < aMin || aValue >= aMax) {
      text = QString().sprintf("%0.2e", aValue);
    } else {
      text = QString().sprintf(aFormat, aValue);
    } // if
    initLineEdit(aEdit, text);
  } // initLineEditF

  int IntOrExc(const QString &aValue) {
    if (aValue == "") {
      return 0;
    } // if
    bool ok = true;
    int i = aValue.toInt(&ok);
    if (ok) {
      return i;
    } else {
      throw std::invalid_argument(qPrintable(QString().sprintf(
	"Not a valid int: \"%s\"", qPrintable(aValue))));
    } // if
  } // IntOrExc

  void reInitCbReaction(QComboBox *aCombo, const QString &aNoneStr) {
    QList <Reaction*> reactions;
    Reaction *reaction;
    /* Sort reactions by output: */
    GerQ::InstContainer *ctr = Reaction::getTheContainer();
    GerQ::DataIterator iter = ctr -> beginData();
    while (iter != ctr -> endData()) {
      Reaction *curReaction = (Reaction*) *iter;
      int i = 0;
      bool found = false;
      while (i < reactions.count() && !found) {
        reaction = reactions[i];
	found = reaction -> getOutput() > curReaction -> getOutput();
	if (!found) {
	  i++;
	} // if
      } // while
      reactions.insert(i, curReaction);
      iter++;
    } // while
    /* Insert sorted reactions into combo: */
    aCombo -> clear();
    aCombo -> addItem(aNoneStr, QVariant());
    while (reactions.count() > 0) {
      reaction = *(reactions.begin());
      aCombo -> addItem(reaction -> getNameGUI(), reaction -> getID());
      reactions.erase(reactions.begin());
    } // while
  } // reInitCbReaction

} // GerFuSp
