/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "ice.h"

#include <gerph/si.h>


namespace GerFuSp {

  GerQ::InstContainer* Ice::fTheContainer = NULL;

  const char* Ice::TABLE                = "TblIce";
  const char* Ice::KEY_ID               = "IDIce";
  const char* Ice::KEY_NAME             = "sName";
  const char* Ice::KEY_FORMULAGUI       = "sFormulaGUI";
  const char* Ice::KEY_FORMULATEX       = "sFormulaTeX";
  const char* Ice::KEY_NUCLEONSREACTIVE = "iNucleonsReactive";
  const char* Ice::KEY_NUCLEONSTOTAL    = "iNucleonsTotal";
  const char* Ice::KEY_DENSITY          = "fDensity_g_cm3";

  Ice::Ice() {
    fNucleonsReactive = -1;
    fNucleonsTotal = -1;
    fDensity = -1.0;
  } // Ice::Ice

  Ice::~Ice() {
    while (fRockets.count() > 0) {
      Rocket *rocket = *(fRockets.begin());
      rocket -> setIce(NULL);
    } // while
  } // Ice::~Ice

  GerQ::DataObject* Ice::doCreate() {
    return new Ice();
  } // Ice::doCreate

  void Ice::addRocket(Rocket *aRocket) {
    QString key = aRocket -> getID().toString();
    if (fRockets.contains(key)) {
      return;
    } // if
    fRockets.insert(key, aRocket);
  } // Ice::addRocket

  void Ice::doRead(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Ice *ice = (Ice*) aData;
    ice -> setName(aSql -> getField(KEY_NAME).toString());
    ice -> setFormulaGUI(aSql -> getField(KEY_FORMULAGUI).toString());
    ice -> setFormulaTeX(aSql -> getField(KEY_FORMULATEX).toString());
    ice -> setNucleonsReactive(aSql -> getField(KEY_NUCLEONSREACTIVE).toInt());
    ice -> setNucleonsTotal(aSql -> getField(KEY_NUCLEONSTOTAL).toInt());
    ice -> setDensity_g_cm3(aSql -> getField(KEY_DENSITY).toDouble());
  } // Ice::doRead

  void Ice::doWrite(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Ice *ice = (Ice*) aData;
    aSql -> setField(KEY_NAME, ice -> getName());
    aSql -> setField(KEY_FORMULAGUI, ice -> getFormulaGUI());
    aSql -> setField(KEY_FORMULATEX, ice -> getFormulaTeX());
    aSql -> setField(KEY_NUCLEONSREACTIVE, ice -> getNucleonsReactive());
    aSql -> setField(KEY_NUCLEONSTOTAL, ice -> getNucleonsTotal());
    aSql -> setField(KEY_DENSITY, ice -> getDensity_g_cm3());
  } // Ice::doWrite

  void Ice::erase() {
    while (fRockets.count() > 0) {
      Rocket *rocket = *(fRockets.begin());
      rocket -> setIce(NULL);
      rocket -> write();
    } // while
    GerQ::DataObject::erase();
  } // Ice::erase

  double Ice::getDensity() {
    return fDensity;
  } // Ice::getDensity

  double Ice::getDensity_g_cm3() {
    return fDensity / GerPh::SI::unit_g_cm3;
  } // Ice::getDensity_g_cm3

  QString Ice::getFormulaGUI() {
    return fFormulaGUI;
  } // Ice::getFormulaGUI

  QString Ice::getFormulaTeX() {
    return fFormulaTeX;
  } // Ice::getFormulaTeX

  QString Ice::getName() {
    return fName;
  } // Ice::getName

  int Ice::getNucleonsReactive() {
    return fNucleonsReactive;
  } // Ice::getNucleonsReactive

  int Ice::getNucleonsTotal() {
    return fNucleonsTotal;
  } // Ice::getNucleonsTotal

  double Ice::getReactiveShare() {
    return (double) fNucleonsReactive / (double) fNucleonsTotal;
  } // Ice::getReactiveShare

  GerQ::InstContainer* Ice::getTheContainer() {
    if (!fTheContainer) {
      fTheContainer = GerQ::DataContainer::addInstDefault(
        TABLE, KEY_ID, doCreate, doRead, doWrite);
      fTheContainer -> getAll();
    } // if
    return fTheContainer;
  } // Ice::getTheContainer

  void Ice::removeRocket(Rocket *aRocket) {
    QString key = aRocket -> getID().toString();
    if (!fRockets.contains(key)) {
      return;
    } // if
    fRockets.remove(key);
  } // Ice::removeRocket

  void Ice::setDensity(double aValue) {
    fDensity = aValue;
  } // Ice::setDensity

  void Ice::setDensity_g_cm3(double aValue) {
    setDensity(aValue * GerPh::SI::unit_g_cm3);
  } // Ice::setDensity_g_cm3

  void Ice::setFormulaGUI(const QString &aValue) {
    fFormulaGUI = aValue;
  } // Ice::setFormulaGUI

  void Ice::setFormulaTeX(const QString &aValue) {
    fFormulaTeX = aValue;
  } // Ice::setFormulaTeX

  void Ice::setName(const QString &aValue) {
    fName = aValue;
  } // Ice::setName

  void Ice::setNucleonsReactive(int aValue) {
    fNucleonsReactive = aValue;
  } // Ice::setNucleonsReactive

  void Ice::setNucleonsTotal(int aValue) {
    fNucleonsTotal = aValue;
  } // Ice::setNucleonsTotal

} // GerFuSp
