/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "mdlice.h"

#include "dlgices.h"
#include "gerfuspbase.h"
#include <QMessageBox>
#include <stdexcept>


namespace GerFuSp {

  MdlIce* MdlIce::fTheModel = NULL;

  const int MdlIce::fColCount = 6;

  const int MdlIce::COL_NAME             = 0;
  const int MdlIce::COL_FORMULAGUI       = 1;
  const int MdlIce::COL_FORMULATEX       = 2;
  const int MdlIce::COL_NUCLEONSREACTIVE = 3;
  const int MdlIce::COL_NUCLEONSTOTAL    = 4;
  const int MdlIce::COL_DENSITY          = 5;

  const char* MdlIce::HDR_NAME             = "Name";
  const char* MdlIce::HDR_FORMULAGUI       = "Formula (GUI)";
  const char* MdlIce::HDR_FORMULATEX       = "Formula (TeX)";
  const char* MdlIce::HDR_NUCLEONSREACTIVE = "Nucleons (react.)";
  const char* MdlIce::HDR_NUCLEONSTOTAL    = "Nucleons (total)";
  const char* MdlIce::HDR_DENSITY          = "Density (g/ccm)";

  MdlIce::MdlIce() {
    readAll();
  } // MdlIce::MdlIce

  void MdlIce::addIce(Ice *aIce) {
    QList <Ice*>::iterator iter = fIces.begin();
    int i = 0;
    bool found = false;
    Ice *ice;
    while (iter != fIces.end() && !found) {
      ice = (*iter);
      found = ice -> getID().toInt() > aIce -> getID().toInt();
      if (!found) {
        i++;
        iter++;
      } // if
    } // while
    fIces.insert(i, aIce);
    emit layoutChanged();
  } // MdlIce::addIce

  int MdlIce::columnCount(const QModelIndex &aParent) const {
    return fColCount;
  } // MdlIce::columnCount

  QVariant MdlIce::data(const QModelIndex &aIndex, int aRole) const {
    if (aRole != Qt::DisplayRole && aRole != Qt::EditRole) {
      return QVariant();
    } // if
    Ice *ice = fIces[aIndex.row()];
    switch (aIndex.column()) {
      case COL_NAME:          return ice -> getName();
      case COL_FORMULAGUI:    return ice -> getFormulaGUI();
      case COL_FORMULATEX:    return ice -> getFormulaTeX();
      case COL_NUCLEONSREACTIVE:
        return QString().sprintf("%i", ice -> getNucleonsReactive());
      case COL_NUCLEONSTOTAL:
        return QString().sprintf("%i", ice -> getNucleonsTotal());
      case COL_DENSITY:
        return QString().sprintf("%0.3f", ice -> getDensity_g_cm3());
    } // switch
    return QVariant();
  } // MdlIce::data

  QModelIndex MdlIce::deleteIce(const QModelIndex &aIndex) {
    if (!aIndex.isValid()) {
      return aIndex;
    } // if
    int row = aIndex.row();
    if (row < 0 || row >= fIces.count()) {
      return aIndex;
    } // if
    Ice *ice = fIces[row];
    if (!ice) {
      return aIndex;
    } // if
    ice -> erase();
    readAll();
    emit layoutChanged();
    int n = fIces.count();
    if (row >= n) {
      row--;
    } // if
    return createIndex(row, aIndex.column());
  } // MdlIce::deleteIce

  Qt::ItemFlags MdlIce::flags(const QModelIndex &aIndex) const {
    Qt::ItemFlags result = QAbstractTableModel::flags(aIndex);
    result |= Qt::ItemIsEditable;
    return result;
  } // MdlIce::flags

  MdlIce* MdlIce::getTheModel() {
    if (!fTheModel) {
      fTheModel = new MdlIce();
    } // if
    return fTheModel;
  } // MdlIce::getTheModel

  QVariant MdlIce::headerData(int aSection, Qt::Orientation aOrientation,
    int aRole
  ) const {
    if (aRole != Qt::DisplayRole) {
      return QVariant();
    } // if
    Ice *ice;
    switch(aOrientation) {
      case Qt::Horizontal:
	switch(aSection) {
	  case COL_NAME:          return HDR_NAME;
	  case COL_FORMULAGUI:    return HDR_FORMULAGUI;
	  case COL_FORMULATEX:    return HDR_FORMULATEX;
	  case COL_NUCLEONSREACTIVE:     return HDR_NUCLEONSREACTIVE;
	  case COL_NUCLEONSTOTAL: return HDR_NUCLEONSTOTAL;
	  case COL_DENSITY:       return HDR_DENSITY;
	  default:                return QVariant();
	} // switch
      case Qt::Vertical:
        ice = fIces[aSection];
	return IDToStr(ice -> getID());
      default: return QVariant();
    } // switch
  } // MdlIce::headerData

  QModelIndex MdlIce::newIce() {
    Ice *ice = (Ice*) Ice::getTheContainer() -> newData();
    ice -> setName(QString().sprintf("Ice %s",
      qPrintable(IDToStr(ice -> getID()))));
    ice -> write();
    readAll();
    int n = fIces.count();
    int i = 0;
    bool found = false;
    while (i < n && !found) {
      found = ice == fIces[i];
      if (!found) {
        i++;
      } // if
    } // while
    return createIndex(i, 0);
  } // MdlIce::newIce

  void MdlIce::readAll() {
    fIces.clear();
    GerQ::InstContainer *ctr = Ice::getTheContainer();
    GerQ::DataIterator iter = ctr -> beginData();
    while (iter != ctr -> endData()) {
      Ice *ice = (Ice*) *iter;
      ice -> read();
      addIce(ice);
      iter++;
    } // while
  } // MdlIce::readAll

  void MdlIce::removeIce(Ice *aIce) {
    QList <Ice*>::iterator iter = fIces.begin();
    Ice *ice;
    while (iter != fIces.end()) {
      ice = (*iter);
      if (ice == aIce) {
        fIces.erase(iter);
      } else {
        iter++;
      } // if
    } // while
  } // MdlIce::removeIce

  int MdlIce::rowCount(const QModelIndex &aParent) const {
    return fIces.count();
  } // MdlIce::rowCount

  bool MdlIce::setData(const QModelIndex &aIndex, const QVariant &aValue,
    int aRole
  ) {
    if (aRole != Qt::EditRole || !aIndex.isValid()) {
      return false;
    } // if
    Ice *ice = fIces[aIndex.row()];
    QString s = aValue.toString();
    try {
      switch (aIndex.column()) {
	case COL_NAME:
	  ice -> setName(s);
	  break;
	case COL_FORMULAGUI:
	  ice -> setFormulaGUI(s);
	  break;
	case COL_FORMULATEX:
	  ice -> setFormulaTeX(s);
	  break;
	case COL_NUCLEONSREACTIVE:
	  ice -> setNucleonsReactive(IntOrExc(s));
	  break;
	case COL_NUCLEONSTOTAL:
	  ice -> setNucleonsTotal(IntOrExc(s));
	  break;
	case COL_DENSITY:
	  ice -> setDensity_g_cm3(DoubleOrExc(s));
	  break;
	default: return false;
      } // switch
    } catch(std::invalid_argument &exc) {
      QMessageBox::information(DlgIces::getTheDialog(), tr("Error"),
        tr(exc.what()));
      return false;
    } // try
    return true;
  } // MdlIce::setData

  void MdlIce::writeAll() {
    QList <Ice*>::iterator iter = fIces.begin();
    Ice *ice;
    while (iter != fIces.end()) {
      ice = *iter;
      ice -> write();
      iter++;
    } // while
    readAll();
  } // MdlIce::writeAll

} // GerFuSp
