/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "mdlreaction.h"

#include "dlgreactions.h"
#include "gerfuspbase.h"
#include <QMessageBox>
#include <stdexcept>


namespace GerFuSp {

  MdlReaction* MdlReaction::fTheModel = NULL;

  const int MdlReaction::fColCount = 4;

  const int MdlReaction::COL_NAMEGUI  = 0;
  const int MdlReaction::COL_NAMETEX  = 1;
  const int MdlReaction::COL_OUTPUT   = 2;
  const int MdlReaction::COL_SHAREGIM = 3;

  const char* MdlReaction::HDR_NAMEGUI  = "Name (GUI)";
  const char* MdlReaction::HDR_NAMETEX  = "Name (TeX)";
  const char* MdlReaction::HDR_OUTPUT   = "Output";
  const char* MdlReaction::HDR_SHAREGIM = "Mass share GIM";

  MdlReaction::MdlReaction() {
    readAll();
  } // MdlReaction::MdlReaction

  void MdlReaction::addReaction(Reaction *aReaction) {
    QList <Reaction*>::iterator iter = fReactions.begin();
    int i = 0;
    bool found = false;
    Reaction *reaction;
    while (iter != fReactions.end() && !found) {
      reaction = (*iter);
      found = reaction -> getOutput() > aReaction -> getOutput();
      if (!found) {
        i++;
        iter++;
      } // if
    } // while
    fReactions.insert(i, aReaction);
    emit layoutChanged();
  } // MdlReaction::addReaction

  int MdlReaction::columnCount(const QModelIndex &aParent) const {
    return fColCount;
  } // MdlReaction::columnCount

  QVariant MdlReaction::data(const QModelIndex &aIndex, int aRole) const {
    if (aRole != Qt::DisplayRole && aRole != Qt::EditRole) {
      return QVariant();
    } // if
    Reaction *reaction = fReactions[aIndex.row()];
    switch (aIndex.column()) {
      case COL_NAMEGUI: return reaction -> getNameGUI();
      case COL_NAMETEX: return reaction -> getNameTeX();
      case COL_OUTPUT:
        return QString().sprintf("%0.6f", reaction -> getOutput());
      case COL_SHAREGIM:
        return QString().sprintf("%0.6f", reaction -> getShareGIM());
    } // switch
    return QVariant();
  } // MdlReaction::data

  QModelIndex MdlReaction::deleteReaction(const QModelIndex &aIndex) {
    if (!aIndex.isValid()) {
      return aIndex;
    } // if
    int row = aIndex.row();
    if (row < 0 || row >= fReactions.count()) {
      return aIndex;
    } // if
    Reaction *reaction = fReactions[row];
    if (!reaction) {
      return aIndex;
    } // if
    reaction -> erase();
    readAll();
    emit layoutChanged();
    int n = fReactions.count();
    if (row >= n) {
      row--;
    } // if
    return createIndex(row, aIndex.column());
  } // MdlReaction::deleteReaction

  Qt::ItemFlags MdlReaction::flags(const QModelIndex &aIndex) const {
    Qt::ItemFlags result = QAbstractTableModel::flags(aIndex);
    result |= Qt::ItemIsEditable;
    return result;
  } // MdlReaction::flags

  MdlReaction* MdlReaction::getTheModel() {
    if (!fTheModel) {
      fTheModel = new MdlReaction();
    } // if
    return fTheModel;
  } // MdlReaction::getTheModel

  QVariant MdlReaction::headerData(int aSection, Qt::Orientation aOrientation,
    int aRole
  ) const {
    if (aRole != Qt::DisplayRole) {
      return QVariant();
    } // if
    Reaction *reaction;
    switch(aOrientation) {
      case Qt::Horizontal:
	switch(aSection) {
	  case COL_NAMEGUI:  return tr(HDR_NAMEGUI);
	  case COL_NAMETEX:  return tr(HDR_NAMETEX);
	  case COL_OUTPUT:   return tr(HDR_OUTPUT);
	  case COL_SHAREGIM: return tr(HDR_SHAREGIM);
	  default:          return QVariant();
	} // switch
      case Qt::Vertical:
        reaction = fReactions[aSection];
	return IDToStr(reaction -> getID());
      default: return QVariant();
    } // switch
  } // MdlReaction::headerData

  QModelIndex MdlReaction::newReaction() {
    Reaction *reaction = (Reaction*) Reaction::getTheContainer() -> newData();
    reaction -> setNameGUI(QString().sprintf("Reaction %s",
      qPrintable(IDToStr(reaction -> getID()))));
    reaction -> write();
    readAll();
    int n = fReactions.count();
    int i = 0;
    bool found = false;
    while (i < n && !found) {
      found = reaction == fReactions[i];
      if (!found) {
        i++;
      } // if
    } // while
    return createIndex(i, 0);
  } // MdlReaction::newReaction

  void MdlReaction::readAll() {
    fReactions.clear();
    GerQ::InstContainer *ctr = Reaction::getTheContainer();
    GerQ::DataIterator iter = ctr -> beginData();
    while (iter != ctr -> endData()) {
      Reaction *reaction = (Reaction*) *iter;
      reaction -> read();
      addReaction(reaction);
      iter++;
    } // while
  } // MdlReaction::readAll

  void MdlReaction::removeReaction(Reaction *aReaction) {
    QList <Reaction*>::iterator iter = fReactions.begin();
    Reaction *reaction;
    while (iter != fReactions.end()) {
      reaction = *iter;
      if (reaction == aReaction) {
        fReactions.erase(iter);
      } else {
        iter++;
      } // if
    } // while
  } // MdlReaction::removeReaction

  int MdlReaction::rowCount(const QModelIndex &aParent) const {
    return fReactions.count();
  } // MdlReaction::rowCount

  bool MdlReaction::setData(const QModelIndex &aIndex, const QVariant &aValue,
    int aRole
  ) {
    if (aRole != Qt::EditRole || !aIndex.isValid()) {
      return false;
    } // if
    Reaction *reaction = fReactions[aIndex.row()];
    QString s = aValue.toString();
    try {
      switch (aIndex.column()) {
	case COL_NAMEGUI:
	  reaction -> setNameGUI(s);
	  break;
	case COL_NAMETEX:
	  reaction -> setNameTeX(s);
	  break;
	case COL_OUTPUT:
	  reaction -> setOutput(DoubleOrExc(s));
	  break;
	case COL_SHAREGIM:
	  reaction -> setShareGIM(DoubleOrExc(s));
	  break;
	default: return false;
      } // switch
    } catch (std::invalid_argument &exc) {
      QMessageBox::information(DlgReactions::getTheDialog(), tr("Error"),
        tr(exc.what()));
      return false;
    } // try
    return true;
  } // MdlReaction::setData

  void MdlReaction::writeAll() {
    QList <Reaction*>::iterator iter = fReactions.begin();
    Reaction *reaction;
    while (iter != fReactions.end()) {
      reaction = *iter;
      reaction -> write();
      iter++;
    } // while
    readAll();
  } // MdlReaction::writeAll

} // GerFuSp
