/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "mdlship.h"


namespace GerFuSp {

  MdlShip* MdlShip::fTheModel = NULL;

  const int MdlShip::fColCount = 10;

  const int MdlShip::COL_NAME       = 0;
  const int MdlShip::COL_TYPE       = 1;
  const int MdlShip::COL_SPECIFICA  = 2;
  const int MdlShip::COL_REACTION   = 3;
  const int MdlShip::COL_OUTPUT     = 4;
  const int MdlShip::COL_REACTSHARE = 5;
  const int MdlShip::COL_EFFICIENCY = 6;
  const int MdlShip::COL_EXHVEL     = 7;
  const int MdlShip::COL_TRAVELSPD  = 8;
  const int MdlShip::COL_COMMENTS   = 9;

  const char* MdlShip::HDR_NAME       = "Name";
  const char* MdlShip::HDR_TYPE       = "Type";
  const char* MdlShip::HDR_SPECIFICA  = "Specifica";
  const char* MdlShip::HDR_REACTION   = "Reaction";
  const char* MdlShip::HDR_OUTPUT     = "Output";
  const char* MdlShip::HDR_REACTSHARE = "Reactive share";
  const char* MdlShip::HDR_EFFICIENCY = "Efficiency";
  const char* MdlShip::HDR_EXHVEL     = "Exhaust vel.";
  const char* MdlShip::HDR_TRAVELSPD  = "Traveling speed";
  const char* MdlShip::HDR_COMMENTS   = "Comments";

  MdlShip::MdlShip() {
    Rocket::getTheContainer();
    Ramjet::getTheContainer();
    GerQ::RefContainer *ctr = Ship::getTheContainer();
    ctr -> getAll();
    GerQ::DataIterator iter = ctr -> beginData();
    while (iter != ctr -> endData()) {
      Ship *ship = (Ship*) *iter;
      addShip(ship);
      iter++;
    } // while
  } // MdlShip::MdlShip

  int MdlShip::addShip(Ship *aShip) {
    int ID = aShip -> getID().toInt();
    QString name = aShip -> getName();
    QList <Ship*>::iterator iter = fShips.end() - 1;
    int i = fShips.count() - 1;
    bool found = false;
    Ship *ship;
    while (i >= 0 && !found) {
      ship = (*iter);
      found =
        ship -> getName() < name ||
        (ship -> getName() == name && ship -> getID().toInt() < ID);
      if (!found) {
        i--;
	iter--;
      } // if
    } // while
    fShips.insert(i + 1, aShip);
    emit layoutChanged();
    return i + 1;
  } // MdlShip::addShip

  int MdlShip::columnCount(const QModelIndex &aParent) const {
    return fColCount;
  } // MdlShip::columnCount

  QVariant MdlShip::data(const QModelIndex &aIndex, int aRole) const {
    if (aRole != Qt::DisplayRole) {
      return QVariant();
    } // if
    int row = aIndex.row();
    int col = aIndex.column();
    Ship *ship = fShips[row];
    if (!ship) {
      return QVariant();
    } // if
    Reaction *reaction = ship -> getReaction();
    double v;
    switch (col) {
      case COL_NAME:       return ship -> getName();
      case COL_TYPE:       return ship -> getTypeName();
      case COL_SPECIFICA:  return ship -> getSpecifica();
      case COL_REACTION:
        if (reaction) {
	  return reaction -> getNameGUI();
	} else {
	  return QVariant();
	} // if
      case COL_OUTPUT:
        if (reaction) {
	  return QString().sprintf("%0.6f", reaction -> getOutput());
	} else {
	  return QVariant();
	} // if
      case COL_REACTSHARE: return QString().sprintf("%0.6f",
        ship -> getReactiveShare());
      case COL_EFFICIENCY: return QString().sprintf("%0.6f",
        ship -> getEfficiency());
      case COL_EXHVEL:
        v = ship -> getExhaustVelocity_km_s();
	if (v < 1000.0) {
	  return QString().sprintf("%0.1fkm/s", v);
	} else {
	  v = ship -> getExhaustVelocity_c();
	  return QString().sprintf("%0.4fc", v);
	} // if
      case COL_TRAVELSPD:
        v = ship -> getTravelSpeed_km_s();
	if (v < 1000.0) {
	  return QString().sprintf("%0.1fkm/s", v);
	} else {
	  v = ship -> getTravelSpeed_c();
	  return QString().sprintf("%0.4fc", v);
	} // if
      case COL_COMMENTS: return ship -> getComments();
    } // switch
    return QVariant();
  } // MdlShip::data

  QModelIndex MdlShip::deleteShip(const QModelIndex &aIndex) {
    if (!aIndex.isValid()) {
      return aIndex;
    } // if
    int row = aIndex.row();
    if (row < 0 || row >= fShips.count()) {
      return aIndex;
    } // if
    Ship *ship = fShips[row];
    if (!ship) {
      return aIndex;
    } // if
    removeShip(ship);
    ship -> erase();
    emit layoutChanged();
    int n = fShips.count();
    if (row >= n) {
      row--;
    } // if
    return createIndex(row, aIndex.column());
  } // MdlShip::deleteShip

  Ship* MdlShip::getShipAt(int aIndex) {
    return fShips[aIndex];
  } // MdlShip::getShipAt

  MdlShip* MdlShip::getTheModel() {
    if (!fTheModel) {
      fTheModel = new MdlShip();
    } // if
    return fTheModel;
  } // MdlShip::getTheModel

  QVariant MdlShip::headerData(int aSection, Qt::Orientation aOrientation,
    int aRole
  ) const {
    if (aRole != Qt::DisplayRole) {
      return QVariant();
    } // if
    Ship *ship;
    switch (aOrientation) {
      case Qt::Horizontal:
        switch (aSection) {
	  case COL_NAME:       return tr(HDR_NAME);
	  case COL_TYPE:       return tr(HDR_TYPE);
	  case COL_SPECIFICA:  return tr(HDR_SPECIFICA);
	  case COL_REACTION:   return tr(HDR_REACTION);
	  case COL_OUTPUT:     return tr(HDR_OUTPUT);
	  case COL_REACTSHARE: return tr(HDR_REACTSHARE);
	  case COL_EFFICIENCY: return tr(HDR_EFFICIENCY);
	  case COL_EXHVEL:     return tr(HDR_EXHVEL);
	  case COL_TRAVELSPD:  return tr(HDR_TRAVELSPD);
	  case COL_COMMENTS:   return tr(HDR_COMMENTS);
	  default:             return QVariant();
	} // switch
      case Qt::Vertical:
        ship = fShips[aSection];
	return IDToStr(ship -> getID());
      default: return QVariant();
    } // switch
  } // MdlShip::headerData

  Ramjet* MdlShip::newRamjet(int &aIndex) {
    Ramjet *ramjet = (Ramjet*) Ramjet::getTheContainer() -> newData();
    ramjet -> setName(QString().sprintf("<Ship %s>",
      qPrintable(IDToStr(ramjet -> getID()))));
    ramjet -> write();
    aIndex = addShip(ramjet);
    return ramjet;
  } // MdlShip::newRamjet

  Rocket* MdlShip::newRocket(int &aIndex) {
    Rocket *rocket = (Rocket*) Rocket::getTheContainer() -> newData();
    rocket -> setName(QString().sprintf("<Ship %s>",
      qPrintable(IDToStr(rocket -> getID()))));
    rocket -> write();
    aIndex = addShip(rocket);
    return rocket;
  } // MdlShip::newRocket

  void MdlShip::removeShip(Ship *aShip) {
    QList <Ship*>::iterator iter = fShips.begin();
    Ship *ship = *iter;
    if (ship == aShip) {
      fShips.removeFirst();
      return;
    } // if
    iter++;
    while (iter != fShips.end() && fShips.count() > 0) {
      ship = *iter;
      if (ship == aShip) {
        fShips.erase(iter);
      } else {
        iter++;
      } // if
    } // while
  } // MdlShip::removeShip

  int MdlShip::rowCount(const QModelIndex &aParent) const {
    return fShips.count();
  } // MdlShip::rowCount

} // GerFuSp
