/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "ramjet.h"

#include <gerph/si.h>
#include <math.h>


namespace GerFuSp {

  GerQ::InstContainer* Ramjet::fTheContainer = NULL;

  const char* Ramjet::TABLE            = "TblRamjet";
  const char* Ramjet::KEY_ID           = "IDShip";
  const char* Ramjet::KEY_INTAKERADIUS = "fIntakeRadius_km";
  const char* Ramjet::KEY_USEDSHARE    = "fUsedShare";

  Ramjet::Ramjet() {
    fIntakeRadius = 1000.0 * GerPh::SI::unit_km;
    fUsedShare = 1.0;
  } // Ramjet::Ramjet

  GerQ::DataObject* Ramjet::doCreate() {
    return new Ramjet();
  } // Ramjet::doCreate

  void Ramjet::doRead(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Ramjet *jet = (Ramjet*) aData;
    jet -> setIntakeRadius_km(aSql -> getField(KEY_INTAKERADIUS).toDouble());
    jet -> setUsedShare(aSql -> getField(KEY_USEDSHARE).toDouble());
  } // Ramjet::doRead

  void Ramjet::doWrite(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Ramjet *jet = (Ramjet*) aData;
    aSql -> setField(KEY_INTAKERADIUS, jet -> getIntakeRadius_km());
    aSql -> setField(KEY_USEDSHARE, jet -> getUsedShare());
  } // Ramjet::doWrite

  double Ramjet::getIntakeRadius() {
    return fIntakeRadius;
  } // Ramjet::getIntakeRadius

  double Ramjet::getIntakeRadius_km() {
    return getIntakeRadius() / GerPh::SI::unit_km;
  } // Ramjet::getIntakeRadius_km

  double Ramjet::getReactiveShare() {
    Reaction *reaction = getReaction();
    if (reaction) {
      return reaction -> getShareGIM() * getUsedShare();
    } else {
      return 0.0;
    } // if
  } // Ramjet::getReactiveShare

  QString Ramjet::getSpecifica() {
    return QString().sprintf("r(A)=%0.0fkm", getIntakeRadius_km());
  } // Ramjet::getSpecifica

  GerQ::InstContainer* Ramjet::getTheContainer() {
    if (!fTheContainer) {
      fTheContainer = GerQ::DataContainer::addInstDefault(TABLE, KEY_ID,
        doCreate, doRead, doWrite, Ship::getTheContainer());
      fTheContainer -> getAll();
    } // if
    return fTheContainer;
  } // Ramjet::getTheContainer

  double Ramjet::getThrust() {
    double m = getStructure();
    double a = getAcceleration();
    double F = m * a;
    return F;
  } // Ramjet::getThrust

  double Ramjet::getTravelSpeed_c() {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return 0.0;
    } // if
    double q = reaction -> getOutput();
    double p = getReactiveShare();
    double eta = getEfficiency();
    double lambda = reaction -> getLorentzFactor(p, eta);
    double v = (1.0 - p * q) * sqrt(pow(lambda, 2.0) - 1.0);
    return v;
  } // Ramjet::getTravelSpeed_c

  QString Ramjet::getTypeName() {
    return "Ramjet";
  } // Ramjet::getTypeName

  double Ramjet::getUsedShare() {
    return fUsedShare;
  } // Ramjet::getUsedShare

  void Ramjet::setIntakeRadius(double aValue) {
    fIntakeRadius = aValue;
  } // Ramjet::setIntakeRadius

  void Ramjet::setIntakeRadius_km(double aValue) {
    setIntakeRadius(aValue * GerPh::SI::unit_km);
  } // Ramjet::setIntakeRadius_km

  void Ramjet::setReactiveShare(double aValue) {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return;
    } // if
    double pg = reaction -> getShareGIM();
    double p = aValue / pg;
    setUsedShare(p);
  } // Ramjet::setReactiveShare

  void Ramjet::setThrust(double aValue) {
    double F = aValue;
    double m = getStructure();
    double a = F / m;
    setAcceleration(a);
  } // Ramjet::setThrust

  void Ramjet::setTravelSpeed_c(double aValue, RecalcRule aRule) {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return;
    } // if
    double q = reaction -> getOutput();
    double p = getReactiveShare();
    double pq = p * q;
    double eta = getEfficiency();
    double v2 = pow(aValue, 2.0);
    double X, Y, Z;
    switch (aRule) {
      case rrByShare:
        X = 1.0 / (2.0 - eta);
	Y = v2 * X / eta;
	Z = X * X - Y;
	if (Z < 0.0) {
	  Z = 0.0;
	} // if
	pq = X - sqrt(Z);
	p = pq / q;
	setReactiveShare(p);
        break;
      case rrByEfficiency:
        X = 1.0 - 1.0 / pq;
	Y = v2 / (pq * pq);
	Z = X * X + Y;
	if (Z < 0.0) {
	  Z = 0.0;
	} // if
	eta = X + sqrt(X * X + Y);
	setEfficiency(eta);
        break;
      default: return;
    } // switch
  } // Ramjet::setTravelSpeed_c

  void Ramjet::setUsedShare(double aValue) {
    double p = aValue;
    if (p < 0.0) {
      p = 0.0;
    } // if
    if (p > 1.0) {
      p = 1.0;
    } // if
    fUsedShare = p;
  } // Ramjet::setUsedShare

} // GerFuSp
