/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "reaction.h"

#include <math.h>
#include <gerph/si.h>


namespace GerFuSp {

  GerQ::InstContainer* Reaction::fTheContainer = NULL;

  const char* Reaction::TABLE        = "TblReaction";
  const char* Reaction::KEY_ID       = "IDReaction";
  const char* Reaction::KEY_NAMEGUI  = "sNameGUI";
  const char* Reaction::KEY_NAMETEX  = "sNameTeX";
  const char* Reaction::KEY_OUTPUT   = "fOutput";
  const char* Reaction::KEY_SHAREGIM = "fShareGIM";

  Reaction::Reaction() {
    fOutput = 0.0;
    fShareGIM = 0.0;
  } // Reaction::Reaction

  Reaction::~Reaction() {
    while (fShips.count() > 0) {
      Ship *ship = *(fShips.begin());
      ship -> setReaction(NULL);
    } // while
  } // Reaction::~Reaction

  void Reaction::addShip(Ship *aShip) {
    QString key = aShip -> getID().toString();
    if (fShips.contains(key)) {
      return;
    } // if
    fShips.insert(key, aShip);
  } // Reaction::addShip

  GerQ::DataObject* Reaction::doCreate() {
    return new Reaction();
  } // Reaction::doCreate

  void Reaction::doRead(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Reaction *reaction = (Reaction*) aData;
    reaction -> setNameGUI(aSql -> getField(KEY_NAMEGUI).toString());
    reaction -> setNameTeX(aSql -> getField(KEY_NAMETEX).toString());
    reaction -> setOutput(aSql -> getField(KEY_OUTPUT).toDouble());
    reaction -> setShareGIM(aSql -> getField(KEY_SHAREGIM).toDouble());
  } // Reaction::doRead

  void Reaction::doWrite(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Reaction *reaction = (Reaction*) aData;
    aSql -> setField(KEY_NAMEGUI, reaction -> getNameGUI());
    aSql -> setField(KEY_NAMETEX, reaction -> getNameTeX());
    aSql -> setField(KEY_OUTPUT, reaction -> getOutput());
    aSql -> setField(KEY_SHAREGIM, reaction -> getShareGIM());
  } // Reaction::doWrite

  void Reaction::erase() {
    while (fShips.count() > 0) {
      Ship *ship = *(fShips.begin());
      ship -> setReaction(NULL);
      ship -> write();
    } // while
    GerQ::DataObject::erase();
  } // Reaction::erase

  double Reaction::getExhaustVelocity(double aReactiveShare,
    double aEfficiency
  ) {
    return getExhaustVelocity_c(aReactiveShare, aEfficiency) *
      GerPh::SI::CONST_c;
  } // Reaction::getExhaustVelocity

  double Reaction::getExhaustVelocity_c(double aReactiveShare,
    double aEfficiency
  ) {
    double lambda = getLorentzFactor(aReactiveShare, aEfficiency);
    double w = sqrt(1.0 - 1.0 / pow(lambda, 2.0));
    return w;
  } // Reaction::getExhaustVelocity_c

  double Reaction::getExhaustVelocity_km_s(double aReactiveShare,
    double aEfficiency
  ) {
    return getExhaustVelocity(aReactiveShare, aEfficiency) /
      GerPh::SI::unit_km_s;
  } // Reaction::getExhaustVelocity_km_s

  double Reaction::getShareGIM() {
    return fShareGIM;
  } // Reaction::getShareGIM

  double Reaction::getLorentzFactor(double aReactiveShare, double aEfficiency)
  {
    double p = aReactiveShare;
    double q = fOutput;
    double eta = aEfficiency;
    double lambda = 1.0 + eta * p * q / (1.0 - p * q);
    return lambda;
  } // Reaction::getLorentzFactor

  QString Reaction::getNameGUI() {
    return fNameGUI;
  } // Reaction::getNameGUI

  QString Reaction::getNamePrinted() {
    if (fNameTeX != "") {
      return fNameTeX;
    } else {
      return fNameGUI;
    } // if
  } // Reaction::getNamePrinted

  QString Reaction::getNameTeX() {
    return fNameTeX;
  } // Reaction::getNameTeX

  double Reaction::getOutput() {
    return fOutput;
  } // Reaction::getOutput

  GerQ::InstContainer* Reaction::getTheContainer() {
    if (!fTheContainer) {
      fTheContainer = GerQ::DataContainer::addInstDefault(
        TABLE, KEY_ID, doCreate, doRead, doWrite);
      fTheContainer -> getAll();
    } // if
    return fTheContainer;
  } // Reaction::getTheContainer

  void Reaction::removeShip(Ship *aShip) {
    QString key = aShip -> getID().toString();
    if (!fShips.contains(key)) {
      return;
    } // if
    fShips.remove(key);
  } // Reaction::removeShip

  void Reaction::setShareGIM(double aValue) {
    fShareGIM = aValue;
  } // Reaction::setShareGIM

  void Reaction::setNameGUI(const QString &aValue) {
    fNameGUI = aValue;
  } // Reaction::setNameGUI

  void Reaction::setNameTeX(const QString &aValue) {
    fNameTeX = aValue;
  } // Reaction::setNameTeX

  void Reaction::setOutput(double aValue) {
    fOutput = aValue;
  } // Reaction::setOutput

} // GerFuSp
