/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "rocket.h"

#include <gerph/si.h>
#include <math.h>

#include <stdio.h> //!!!


#define APPROACH_MU(GET_DISTANCE_NOW) \
    double delta; \
    const double eps = 1.0e-8; \
    do { \
      setConsumption(mu); \
      delta = (GET_DISTANCE_NOW() - s) * _s; \
      if (fabs(delta) > eps) { \
        mu = mu * (1.0 + delta); \
      } \
    } while (fabs(delta) > eps);
// APPROACH_MU


#define APPROACH_T_INERTIAL(M_PHASE, T_OB_PHASE) \
    double m0 = getStructure(); \
    double F = getThrust(); \
    double c = GerPh::SI::CONST_c; \
    double mu = getConsumption(); \
    double R = getRatioFullEmpty(); \
    double sqrt_R = sqrt(R); \
    double m = M_PHASE; \
    double A = m / (F / c + c * mu * mu / F); \
    double mu_m = mu / m; \
    double F_cmu = F / (c * mu); \
    double cmu_F = 1.0 / F_cmu; \
    double v = getTravelSpeed_c(); \
    double T_ob_correct = T_OB_PHASE; \
    double T_lo = T_ob_correct; \
    double T_hi = T_ob_correct / sqrt(1.0 - v * v); \
    const double eps = 1.0e-8; \
    double X, a_ln_b, T_ob; \
    double T = 0.5 * (T_hi + T_lo); \
    do { \
      X = 1.0 - mu_m * T; \
      a_ln_b = F_cmu * log(X); \
      T_ob = A * (1.0 - X * (sin(a_ln_b) + cmu_F * cos(a_ln_b))); \
      if (T_ob > T_ob_correct) { \
	T_hi = T; \
      } else { \
	T_lo = T; \
      } \
      T = 0.5 * (T_lo + T_hi); \
    } while (fabs((T_hi - T_lo) / T_lo) > eps); \
    return T;
// APPROACH_T_INERTIAL


namespace GerFuSp {

  GerQ::InstContainer* Rocket::fTheContainer = NULL;

  const char* Rocket::TABLE              = "TblRocket";
  const char* Rocket::KEY_ID             = "IDShip";
  const char* Rocket::KEY_ADDPROPPERFUEL = "fAddPropPerFuel";
  const char* Rocket::KEY_RATIOFULLEMPTY = "fRatioFullEmpty";
  const char* Rocket::KEY_ICE            = "IDIce";

  Rocket::Rocket() {
    fAddPropPerFuel = 0.0;
    fRatioFullEmpty = 1.0;
    fIce = NULL;
  } // Rocket::Rocket

  Rocket::~Rocket() {
    if (fIce) {
      fIce -> removeRocket(this);
    } // if
  } // Rocket::~Rocket

  GerQ::DataObject* Rocket::doCreate() {
    return new Rocket();
  } // Rocket::doCreate

  void Rocket::doRead(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Rocket *rocket = (Rocket*) aData;
    rocket -> setAddPropPerFuel(
      aSql -> getField(KEY_ADDPROPPERFUEL).toDouble());
    rocket -> setRatioFullEmpty(
      aSql -> getField(KEY_RATIOFULLEMPTY).toDouble());
    rocket -> setIce((Ice*) GerQ::DataContainer::getDataDefault(
      Ice::getTheContainer(), aSql, KEY_ICE));
  } // Rocket::doRead

  void Rocket::doWrite(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Rocket *rocket = (Rocket*) aData;
    aSql -> setField(KEY_ADDPROPPERFUEL, rocket -> getAddPropPerFuel());
    aSql -> setField(KEY_RATIOFULLEMPTY, rocket -> getRatioFullEmpty());
    GerQ::DataContainer::setDataDefault(aSql, KEY_ICE, rocket -> getIce());
  } // Rocket::doWrite

  double Rocket::getAccelDistance() {
    double R = getRatioFullEmpty();
    return sqrt(R) * getDecelDistance();
  } // Rocket::getAccelDistance

  double Rocket::getAccelDistance_ly() {
    return getAccelDistance() / GerPh::SI::unit_ly;
  } // Rocket::getAccelDistance_ly

  double Rocket::getAccelDistance_km() {
    return getAccelDistance() / GerPh::SI::unit_km;
  } // Rocket::getAccelDistance_km

  double Rocket::getAccelEmpty() {
    return getAcceleration() * getRatioFullEmpty();
  } // Rocket::getAccelEmpty

  double Rocket::getAccelEmpty_g() {
    return getAccelEmpty() / GerPh::SI::CONST_g;
  } // Rocket::getAccelEmpty_g

  double Rocket::getAccelFull() {
    return getAcceleration();
  } // Rocket::getAccelFull

  double Rocket::getAccelFull_g() {
    return getAccelFull() / GerPh::SI::CONST_g;
  } // Rocket::getAccelFull_g

  double Rocket::getAccelTimeInertial() {
    APPROACH_T_INERTIAL(R * m0, m0 * (R - sqrt_R) / mu);
  } // Rocket::getAccelTimeInertial

  double Rocket::getAccelTimeOnBoard() {
    double R = getRatioFullEmpty();
    return sqrt(R) * getDecelTimeOnBoard();
  } // Rocket::getAccelTimeOnBoard

  double Rocket::getAddPropPerFuel() {
    return fAddPropPerFuel;
  } // Rocket::getAddPropPerFuel

  double Rocket::getDecelDistance() {
    double m0 = getStructure();
    double F = getThrust();
    double c = GerPh::SI::CONST_c;
    double F_c = F / c;
    double mu = getConsumption();
    double R = getRatioFullEmpty();
    double X = m0 * F / (F_c * F_c + mu * mu);
    double Y = log(R) * F / (2.0 * c * mu);
    double T = X * (sqrt(R) - (cos(Y) + sin(Y) * c * mu / F));
    return T;
  } // Rocket::getDecelDistance

  double Rocket::getDecelDistance_ly() {
    return getDecelDistance() / GerPh::SI::unit_ly;
  } // Rocket::getDecelDistance_ly

  double Rocket::getDecelDistance_km() {
    return getDecelDistance() / GerPh::SI::unit_km;
  } // Rocket::getDecelDistance_km

  double Rocket::getDecelTimeInertial() {
    APPROACH_T_INERTIAL(sqrt_R * m0, m0 * (sqrt_R - 1.0) / mu);
  } // Rocket::getDecelTimeInertial

  double Rocket::getDecelTimeOnBoard() {
    double R = getRatioFullEmpty();
    double m0 = getStructure();
    double mu = getConsumption();
    double T = (sqrt(R) - 1.0) * m0 / mu;
    return T;
  } // Rocket::getDecelTimeOnBoard

  double Rocket::getFuelMass() {
    return getStartingMass() - getStructure();
  } // Rocket::getFuelMass

  double Rocket::getFuelMass_t() {
    return getFuelMass() / GerPh::SI::unit_t;
  } // Rocket::getFuelMass_t

  Ice* Rocket::getIce() {
    return fIce;
  } // Rocket::getIce

  double Rocket::getIceDiameter() {
    Ice *ice = getIce();
    if (!ice) {
      return -1.0;
    } // if
    double m = getFuelMass();
    double rho = ice -> getDensity();
    double V = m / rho;
    double pi = GerPh::SI::CONST_pi;
    double r = pow(0.75 * V / pi, 1.0 / 3.0);
    return 2.0 * r;
  } // Rocket::getIceDiameter

  double Rocket::getIceDiameter_km() {
    return getIceDiameter() / GerPh::SI::unit_km;
  } // Rocket::getIceDiameter_km

  double Rocket::getRatioFullEmpty() {
    return fRatioFullEmpty;
  } // Rocket::getRatioFullEmpty

  double Rocket::getReactiveShare() {
    double J = fAddPropPerFuel;
    double p = 1.0 / (1.0 + J);
    if (fIce) {
      p *= fIce -> getReactiveShare();
    } // if
    return p;
  } // Rocket::getReactiveShare

  QString Rocket::getSpecifica() {
    QString result;
    if (fIce) {
      result = QString().sprintf("R=%0.2e (%s)", fRatioFullEmpty,
        qPrintable(fIce -> getFormulaGUI()));
    } else {
      result = QString().sprintf("R=%0.1f", fRatioFullEmpty);
    } // if
    return result;
  } // Rocket::getSpecifica

  double Rocket::getStartingMass() {
    double m = getStructure();
    double R = getRatioFullEmpty();
    return R * m;
  } // Rocket::getStartingMass

  double Rocket::getStartingMass_t() {
    return getStartingMass() / GerPh::SI::unit_t;
  } // Rocket::getStartingMass_t

  GerQ::InstContainer* Rocket::getTheContainer() {
    if (!fTheContainer) {
      fTheContainer = GerQ::DataContainer::addInstDefault(TABLE, KEY_ID,
        doCreate, doRead, doWrite, Ship::getTheContainer());
      fTheContainer -> getAll();
    } // if
    return fTheContainer;
  } // Rocket::getTheContainer

  double Rocket::getThrust() {
    double a = getAccelFull();
    double m = getStartingMass();
    double F = m * a;
    return F;
  } // Rocket::getThrust

  double Rocket::getTotalFiringDistance() {
    double R = getRatioFullEmpty();
    return (1.0 + sqrt(R)) * getDecelDistance();
  } // Rocket::getTotalFiringDistance

  double Rocket::getTotalFiringDistance_ly() {
    return getTotalFiringDistance() / GerPh::SI::unit_ly;
  } // Rocket::getTotalFiringDistance_ly

  double Rocket::getTotalFiringDistance_km() {
    return getTotalFiringDistance() / GerPh::SI::unit_km;
  } // Rocket::getTotalFiringDistance_km

  double Rocket::getTotalFiringTimeInertial() {
    return getAccelTimeInertial() + getDecelTimeInertial();
  } // Rocket::getTotalFiringTimeInertial

  double Rocket::getTotalFiringTimeOnBoard() {
    double R = getRatioFullEmpty();
    double m0 = getStructure();
    double mf = (R - 1.0) * m0;
    double mu = getConsumption();
    double T = mf / mu;
    return T;
  } // Rocket::getTotalFiringTimeOnBoard

  double Rocket::getTravelSpeed_c() {
    if (!getReaction()) {
      return 0.0;
    } // if
    double R = fRatioFullEmpty;
    double w_c = getExhaustVelocity_c();
    double X = pow(R, w_c);
    return (X - 1.0) / (X + 1.0);
  } // Rocket::getTravelSpeed_c

  QString Rocket::getTypeName() {
    if (fIce) {
      return "Ice Rocket";
    } else {
      return "Closed Rocket";
    } // if
  } // Rocket::getTypeName

  void Rocket::setAccelDistance(double aValue) {
    if (aValue < 1.0e-10) {
      return;
    } // if
    double m0 = getStructure();
    double I = getSpecificImpulse();
    double s = aValue;
    double _s = 1.0 / s;
    double R = getRatioFullEmpty();
    double sqrt_R = sqrt(R);
    double A = m0 * I * _s;
    double mu = A * sqrt_R * (sqrt_R - (1.0 + 0.5 * log(R)));
    APPROACH_MU(getAccelDistance);
  } // Rocket::setAccelDistance

  void Rocket::setAccelDistance_km(double aValue) {
    setAccelDistance(aValue * GerPh::SI::unit_km);
  } // Rocket::setAccelDistance_km

  void Rocket::setAccelDistance_ly(double aValue) {
    setAccelDistance(aValue * GerPh::SI::unit_ly);
  } // Rocket::setAccelDistance_ly

  void Rocket::setAccelEmpty(double aValue) {
    double R = getRatioFullEmpty();
    if (abs(R) > 1.0e-20) {
      setAcceleration(aValue / R);
    } // if
  } // Rocket::setAccelEmpty

  void Rocket::setAccelEmpty_g(double aValue) {
    setAccelEmpty(aValue * GerPh::SI::CONST_g);
  } // Rocket::setAccelEmpty

  void Rocket::setAccelFull(double aValue) {
    if (aValue > 0.0) {
      setAcceleration(aValue);
    } else {
      setAcceleration(0.0);
    } // if
  } // Rocket::setAccelFull

  void Rocket::setAccelFull_g(double aValue) {
    setAccelFull(aValue * GerPh::SI::CONST_g);
  } // Rocket::setAccelFull_g

  void Rocket::setAddPropPerFuel(double aValue) {
    if (aValue > 0.0) {
      fAddPropPerFuel = aValue;
    } else {
      fAddPropPerFuel = 0.0;
    } // if
  } // Rocket::setAddPropPerFuel

  void Rocket::setDecelDistance(double aValue) {
    if (aValue < 1.0e-10) {
      return;
    } // if
    double m0 = getStructure();
    double I = getSpecificImpulse();
    double s = aValue;
    double _s = 1.0 / s;
    double R = getRatioFullEmpty();
    double sqrt_R = sqrt(R);
    double A = m0 * I * _s;
    double mu = A * (sqrt_R - (1.0 + 0.5 * log(R)));
    APPROACH_MU(getDecelDistance);
  } // Rocket::setDecelDistance

  void Rocket::setDecelDistance_km(double aValue) {
    setDecelDistance(aValue * GerPh::SI::unit_km);
  } // Rocket::setDecelDistance_km

  void Rocket::setDecelDistance_ly(double aValue) {
    setDecelDistance(aValue * GerPh::SI::unit_ly);
  } // Rocket::setDecelDistance_ly

  void Rocket::setFuelMass(double aValue) {
    if (aValue > 0.0) {
      setStartingMass(aValue + getStructure());
    } else {
      setStartingMass(getStructure());
    } // if
  } // Rocket::setFuelMass

  void Rocket::setFuelMass_t(double aValue) {
    setFuelMass(aValue * GerPh::SI::unit_t);
  } // Rocket::setFuelMass_t

  void Rocket::setIce(Ice *aValue) {
    if (fIce) {
      fIce -> removeRocket(this);
    } // if
    fIce = aValue;
    if (fIce) {
      fIce -> addRocket(this);
    } // if
  } // Rocket::setIce

  void Rocket::setIceDiameter(double aValue) {
    Ice *ice = getIce();
    if (!ice) {
      return;
    } // if
    if (aValue > 0.0) {
      double r = 0.5 * aValue;
      double pi = GerPh::SI::CONST_pi;
      double V = (4.0 * pi * pow(r, 3.0)) / 3.0;
      double rho = ice -> getDensity();
      double m = rho * V;
      setFuelMass(m);
    } else {
      setFuelMass(0.0);
    } // if
  } // Rocket::setIceDiameter

  void Rocket::setIceDiameter_km(double aValue) {
    setIceDiameter(aValue * GerPh::SI::unit_km);
  } // Rocket::setIceDiameter_km

  void Rocket::setRatioFullEmpty(double aValue) {
    if (aValue > 1.0) {
      fRatioFullEmpty = aValue;
    } else {
      fRatioFullEmpty = 1.0;
    } // if
  } // Rocket::setRatioFullEmpty

  void Rocket::setReactiveShare(double aValue) {
    double p = 1.0;
    if (fIce) {
      p = fIce -> getReactiveShare();
    } // if
    double J = p / aValue - 1.0;
    if (J < 0.0) {
      J = 0.0;
    } // if
    fAddPropPerFuel = J;
  } // Rocket::setReactiveShare

  void Rocket::setStartingMass(double aValue) {
    double m = getStructure();
    if (abs(m) < 1.0e-20) {
      return;
    } // if
    setRatioFullEmpty(aValue / m);
  } // Rocket::setStartingMass

  void Rocket::setStartingMass_t(double aValue) {
    setStartingMass(aValue * GerPh::SI::unit_t);
  } // Rocket::setStartingMass_t

  void Rocket::setThrust(double aValue) {
    double F = aValue;
    double m = getStartingMass();
    double a = F / m;
    setAccelFull(a);
  } // Rocket::setThrust

  void Rocket::setTotalFiringDistance(double aValue) {
    if (aValue < 1.0e-10) {
      return;
    } // if
    double m0 = getStructure();
    double I = getSpecificImpulse();
    double s = aValue;
    double _s = 1.0 / s;
    double R = getRatioFullEmpty();
    double sqrt_R = sqrt(R);
    double A = m0 * I * _s;
    double mu = A * (sqrt_R + 1.0) * (sqrt_R - (1.0 + 0.5 * log(R)));
    APPROACH_MU(getTotalFiringDistance);
  } // Rocket::setTotalFiringDistance

  void Rocket::setTotalFiringDistance_km(double aValue) {
    setTotalFiringDistance(aValue * GerPh::SI::unit_km);
  } // Rocket::setTotalFiringDistance_km

  void Rocket::setTotalFiringDistance_ly(double aValue) {
    setTotalFiringDistance(aValue * GerPh::SI::unit_ly);
  } // Rocket::setTotalFiringDistance_ly

  void Rocket::setTravelSpeed_c(double aValue, RecalcRule aRule) {
    double v = aValue;
    if (v < 0.0 || v >= 1.0) {
      return;
    } // if
    double R = getRatioFullEmpty();
    double w = getExhaustVelocity_c();
    switch (aRule) {
      case rrByFuel:
        R = pow((1.0 + v) / (1.0 - v), 1.0 / w);
	setRatioFullEmpty(R);
        break;
      default:
        w = log((1.0 + v) / (1.0 - v)) / log(R);
        setExhaustVelocity_c(w, aRule);
        break;
    } // switch
  } // Rocket::setTravelSpeed_c

} // GerFuSp
