/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "ship.h"

#include <gerph/si.h>
#include <math.h>


namespace GerFuSp {

  GerQ::RefContainer* Ship::fTheContainer = NULL;

  const char* Ship::TABLE            = "TblShip";
  const char* Ship::KEY_ID           = "IDShip";
  const char* Ship::KEY_NAME         = "sName";
  const char* Ship::KEY_REACTION     = "IDReaction";
  const char* Ship::KEY_EFFICIENCY   = "fEfficiency";
  const char* Ship::KEY_STRUCTURE    = "fStructure_t";
  const char* Ship::KEY_ACCELERATION = "fAcceleration_g";
  const char* Ship::KEY_COMMENTS     = "sComments";

  Ship::Ship() {
    fReaction = NULL;
    fEfficiency = 1.0;
    fStructure = 1000.0 * GerPh::SI::unit_t;
    fAcceleration = 0.01 * GerPh::SI::CONST_g;
  } // Ship::Ship

  Ship::~Ship() {
    if (fReaction) {
      fReaction -> removeShip(this);
    } // if
  } // Ship::~Ship

  void Ship::doRead(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Ship *ship = (Ship*) aData;
    ship -> setName(aSql -> getField(KEY_NAME).toString());
    ship -> setReaction((Reaction*) GerQ::DataContainer::getDataDefault(
      Reaction::getTheContainer(), aSql, KEY_REACTION));
    ship -> setEfficiency(aSql -> getField(KEY_EFFICIENCY).toDouble());
    ship -> setStructure_t(aSql -> getField(KEY_STRUCTURE).toDouble());
    ship -> setAcceleration_g(aSql -> getField(KEY_ACCELERATION).toDouble());
    ship -> setComments(aSql -> getField(KEY_COMMENTS).toString());
  } // Ship::doRead

  void Ship::doWrite(GerQ::DataObject *aData, GerQ::SqlDataAdapter *aSql) {
    Ship *ship = (Ship*) aData;
    aSql -> setField(KEY_NAME, ship -> getName());
    GerQ::DataContainer::setDataDefault(aSql, KEY_REACTION,
      ship -> getReaction());
    aSql -> setField(KEY_EFFICIENCY, ship -> getEfficiency());
    aSql -> setField(KEY_STRUCTURE, ship -> getStructure_t());
    aSql -> setField(KEY_ACCELERATION, ship -> getAcceleration_g());
    aSql -> setField(KEY_COMMENTS, ship -> getComments());
  } // Ship::doWrite

  double Ship::getAcceleration() {
    return fAcceleration;
  } // Ship::getAcceleration

  double Ship::getAcceleration_g() {
    return getAcceleration() / GerPh::SI::CONST_g;
  } // Ship::getAcceleration_g

  QString Ship::getComments() {
    return fComments;
  } // Ship::getComments

  double Ship::getConsumption() {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return 0.0;
    } // if
    double I = getSpecificImpulse();
/*!!!
    double pq = reaction -> getOutput() * getReactiveShare();
    double eta = getEfficiency();
    double F = getThrust();
    double c = GerPh::SI::CONST_c;
    double mu = F / (c * sqrt(eta * pq * (2.0 - (2.0 - eta) * pq)));
!!!*/
    double F = getThrust();
    double mu = F / I;
    return mu;
  } // Ship::getConsumption

  double Ship::getConsumption_kg_s() {
    return getConsumption() / GerPh::SI::unit_kg_s;
  } // Ship::getConsumption_kg_s

  double Ship::getDissipation() {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return 0.0;
    } // if
    double mu = getConsumption();
    double c = GerPh::SI::CONST_c;
    double c2 = c * c;
    double eta = getEfficiency();
    double pq = reaction -> getOutput() * getReactiveShare();
    double P_d = mu * c2 * (1.0 - eta) * pq;
    return P_d;
  } // Ship::getDissipation

  double Ship::getDissipation_W() {
    return getDissipation() / GerPh::SI::unit_kg_s;
  } // Ship::getDissipation_W

  double Ship::getEfficiency() {
    return fEfficiency;
  } // Ship::getEfficiency

  double Ship::getExhaustVelocity() {
    return getExhaustVelocity_c() * GerPh::SI::CONST_c;
  } // Ship::getExhaustVelocity

  double Ship::getExhaustVelocity_c() {
    if (fReaction && fEfficiency > 0.0) {
      double p = getReactiveShare();
      double eta = fEfficiency;
      return fReaction -> getExhaustVelocity_c(p, eta);
    } else {
      return 0.0;
    } // 
  } // Ship::getExhaustVelocity_c

  double Ship::getExhaustVelocity_km_s() {
    return getExhaustVelocity() / GerPh::SI::unit_km_s;
  } // Ship::getExhaustVelocity_km_s

  QString Ship::getName() {
    return fName;
  } // Ship::getName

  double Ship::getPower() {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return 0.0;
    } // if
    double mu = getConsumption();
    double c = GerPh::SI::CONST_c;
    double c2 = c * c;
    double eta = getEfficiency();
    double pq = reaction -> getOutput() * getReactiveShare();
    double P = mu * c2 * eta * pq;
    return P;
  } // Ship::getPower

  double Ship::getPower_W() {
    return getPower() / GerPh::SI::unit_W;
  } // Ship::getPower_W

  Reaction* Ship::getReaction() {
    return fReaction;
  } // Ship::getReaction

  double Ship::getSpecificImpulse() {
    return getSpecificImpulse_c() * GerPh::SI::CONST_c;
  } // Ship::getSpecificImpulse

  double Ship::getSpecificImpulse_c() {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return 0.0;
    } // if
    double pq = reaction -> getOutput() * getReactiveShare();
    double eta = getEfficiency();
    double I = sqrt(eta * pq * (2.0 - (2.0 - eta) * pq));
    return I;
  } // Ship::getSpecificImpulse_c

  double Ship::getSpecificImpulse_km_s() {
    return getSpecificImpulse() / GerPh::SI::unit_km_s;
  } // Ship::getSpecificImpulse_km_s

  double Ship::getStructure() {
    return fStructure;
  } // Ship::getStructure

  double Ship::getStructure_t() {
    return getStructure() / GerPh::SI::unit_t;
  } // Ship::getStructure_t

  GerQ::RefContainer* Ship::getTheContainer() {
    if (!fTheContainer) {
      fTheContainer = GerQ::DataContainer::addRefDefault(
        TABLE, KEY_ID, doRead, doWrite);
    } // if
    return fTheContainer;
  } // Ship::getTheContainer

  double Ship::getThrust_N() {
    return getThrust() / GerPh::SI::unit_N;
  } // Ship::getThrust_N

  double Ship::getTravelSpeed() {
    return getTravelSpeed_c() * GerPh::SI::CONST_c;
  } // Ship::getTravelSpeed

  double Ship::getTravelSpeed_km_s() {
    return getTravelSpeed() / GerPh::SI::unit_km_s;
  } // Ship::getTravelSpeed_km_s

  void Ship::setAcceleration(double aValue) {
    if (aValue > 0.0) {
      fAcceleration = aValue;
    } else {
      fAcceleration = 0.0;
    } // if
  } // Ship::setAcceleration

  void Ship::setAcceleration_g(double aValue) {
    setAcceleration(aValue * GerPh::SI::CONST_g);
  } // Ship::setAcceleration_g

  void Ship::setComments(const QString &aValue) {
    fComments = aValue;
  } // Ship::setComments

  void Ship::setConsumption(double aValue) {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return;
    } // if
    double pq = reaction -> getOutput() * getReactiveShare();
    double eta = getEfficiency();
    double c = GerPh::SI::CONST_c;
    double mu = aValue;
    double F = mu * c * sqrt(eta * pq * (2.0 - (2.0 - eta) * pq));
    setThrust(F);
  } // Ship::setConsumption

  void Ship::setConsumption_kg_s(double aValue) {
    setConsumption(aValue * GerPh::SI::unit_kg_s);
  } // Ship::setConsumption_kg_s

  void Ship::setDissipation(double aValue, RecalcRule aRule) {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return;
    } // if
    double P = aValue;
    double P2 = P * P;
    double F = getThrust();
    double c = GerPh::SI::CONST_c;
    double c2 = c * c;
    double Fc2 = F * F * c2;
    double p = getReactiveShare();
    double q = reaction -> getOutput();
    double pq = p * q;
    double eta = getEfficiency();
    double A, B, mu;
    switch (aRule) {
      case rrByShare:
        p = 2.0 * P2 * eta /
	    ((Fc2 * (1.0 - eta) * (1.0 - eta) + P2 * eta * (2.0 - eta)) * q);
	setReactiveShare(p);
        break;
      case rrByEfficiency:
        A = (Fc2 * pq + P2 * (1.0 - pq)) / ((Fc2 - P2) * pq);
	B = Fc2 / (Fc2 - P2);
	eta = A - sqrt(A * A - B);
        setEfficiency(eta);
        break;
      case rrByFuel:
        mu = P / (c2 * (1.0 - eta) * pq);
	setConsumption(mu);
        break;
    } // switch
  } // Ship::setDissipation

  void Ship::setDissipation_W(double aValue, RecalcRule aRule) {
    setDissipation(aValue * GerPh::SI::unit_W, aRule);
  } // Ship::setDissipation_W

  void Ship::setEfficiency(double aValue) {
    if (aValue < 0.0) {
      fEfficiency = 0.0;
    } else if (aValue > 1.0) {
      fEfficiency = 1.0;
    } else {
      fEfficiency = aValue;
    } // if
  } // Ship::setEfficiency

  void Ship::setExhaustVelocity(double aValue, RecalcRule aRule) {
    setExhaustVelocity_c(aValue / GerPh::SI::CONST_c, aRule);
  } // Ship::setExhaustVelocity

  void Ship::setExhaustVelocity_c(double aValue, RecalcRule aRule) {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return;
    } // if
    if (aValue < 0.0 || aValue >= 1.0) {
      return;
    } // if
    double q = reaction -> getOutput();
    double w = aValue;
    double lambda = 1.0 / sqrt(1.0 - w * w);
    double p, eta;
    switch (aRule) {
      case rrByShare:
        eta = getEfficiency();
	p = (lambda - 1.0) / (q * (lambda - 1.0 + eta));
	setReactiveShare(p);
        break;
      case rrByEfficiency:
        p = getReactiveShare();
	eta = (lambda - 1.0) * (1.0 - p * q) / (p * q);
	setEfficiency(eta);
        break;
      default:
        return;
    } // switch
  } // Ship::setExhaustVelocity_c

  void Ship::setExhaustVelocity_km_s(double aValue, RecalcRule aRule) {
    setExhaustVelocity(aValue * GerPh::SI::unit_km_s, aRule);
  } // Ship::setExhaustVelocity_km_s

  void Ship::setName(const QString &aValue) {
    fName = aValue;
  } // Ship::setName

  void Ship::setPower(double aValue, RecalcRule aRule) {
    Reaction *reaction = getReaction();
    if (!reaction) {
      return;
    } // if
    double P = aValue;
    double P2 = P * P;
    double F = getThrust();
    double c = GerPh::SI::CONST_c;
    double c2 = c * c;
    double Fc2 = F * F * c2;
    double eta = getEfficiency();
    double p = getReactiveShare();
    double q = reaction -> getOutput();
    double pq = p * q;
    double mu = getConsumption();
    switch (aRule) {
      case rrByShare:
        p = 2.0 * P2 / ((Fc2 * eta + (2.0 - eta) * P2) * q);
	setReactiveShare(p);
        break;
      case rrByEfficiency:
        eta = 2.0 * P2 * (1.0 - pq) / ((Fc2 - P2) * pq);
	setEfficiency(eta);
        break;
      case rrByFuel:
        mu = P / (c2 * eta * pq);
        setConsumption(mu);
        break;
    } // switch
  } // Ship::setPower

  void Ship::setPower_W(double aValue, RecalcRule aRule) {
    setPower(aValue * GerPh::SI::unit_W, aRule);
  } // Ship::setPower_W

  void Ship::setReaction(Reaction *aValue) {
    if (fReaction) {
      fReaction -> removeShip(this);
    } // if
    fReaction = aValue;
    if (fReaction) {
      fReaction -> addShip(this);
    } // if
  } // Ship::setReaction

  void Ship::setStructure(double aValue) {
    if (aValue > 0.0) {
      fStructure = aValue;
    } else {
      fStructure = 0.0;
    } // if
  } // Ship::setStructure

  void Ship::setStructure_t(double aValue) {
    setStructure(aValue * GerPh::SI::unit_t);
  } // Ship::setStructure_t

  void Ship::setThrust_N(double aValue) {
    setThrust(aValue * GerPh::SI::unit_N);
  } // Ship::setThrust_N

  void Ship::setTravelSpeed(double aValue, RecalcRule aRule) {
    setTravelSpeed_c(aValue / GerPh::SI::CONST_c, aRule);
  } // Ship::setTravelSpeed

  void Ship::setTravelSpeed_km_s(double aValue, RecalcRule aRule) {
    setTravelSpeed(aValue * GerPh::SI::unit_km_s, aRule);
  } // Ship::setTravelSpeed_km_s

} // GerFuSp
