/*
  This file is part of GerFuSp.

  GerFuSp is free software: you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  GerFuSp is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  GerFuSp.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "wndmain.h"

#include "dlgices.h"
#include "dlgramjet.h"
#include "dlgreactions.h"
#include "dlgrocket.h"
#include "mdlship.h"
#include "ramjet.h"
#include "rocket.h"
#include <QHeaderView>
#include <QIcon>
#include <QMenuBar>
#include <QMessageBox>


namespace GerFuSp {

  WndMain* WndMain::fTheWindow = NULL;

  const char* WndMain::TITLE = "Ger's Futuristic Spacecraft Calculator";
  const char* WndMain::ICON = "/usr/share/gerfusp/icon-wndmain.png";

  WndMain::WndMain() {
    fMnFile = NULL;
    fMnNewShip = NULL;
    fActNewRocket = NULL;
    fActNewRamjet = NULL;
    fActEditShip = NULL;
    fActDeleteShip = NULL;
    fSepQuit = NULL;
    fActQuit = NULL;
    fMnOptions = NULL;
    fActReactions = NULL;
    fActIces = NULL;
    fWgtMain = NULL;
    fBoxMain = NULL;
    fLbShips = NULL;
    fTblShips = NULL;
    fCtxMnShips = NULL;
    fCtxActEditShip = NULL;
    fCtxActDeleteShip = NULL;
    buildMenu();
    setCentralWidget(getWgtMain());
    setWindowTitle(tr(TITLE));
    setWindowIcon(QIcon(ICON));
    resize(800, 600);
  } // WndMain::WndMain

  void WndMain::buildMenu() {
    getMnFile();
    getMnOptions();
  } // WndMain::buildMenu

  void WndMain::closeEvent(QCloseEvent *aEvent) {
    doQuit();
  } // WndMain::closeEvent

  void WndMain::doCtxMnShips(const QPoint &aPoint) {
    getCtxMnShips() -> popup(getTblShips() -> viewport() ->
      mapToGlobal(aPoint));
  } // WndMain::doCtxMnShips

  void WndMain::doDeleteShip() {
    QTableView *table = getTblShips();
    MdlShip *model = MdlShip::getTheModel();
    int row = curTableRow(table);
    Ship *ship = model -> getShipAt(row);
    int response = QMessageBox::question(
      this,
      tr("Question"),
      tr(qPrintable(
        QString().sprintf(
          "Please confirm: Delete ship %s, \"%s\"?",
          qPrintable(IDToStr(ship -> getID())),
	  qPrintable(ship -> getName())
      ))),
      QMessageBox::Yes,
      QMessageBox::No | QMessageBox::Default);
    if (response != QMessageBox::Yes) {
      return;
    } // if
    dropShipDialog(ship);
    QModelIndex idx = table -> currentIndex();
    table -> setCurrentIndex(model -> deleteShip(idx));
  } // WndMain::doDeleteShip

  void WndMain::doEditShip() {
    int row = curTableRow(getTblShips());
    Ship *ship = MdlShip::getTheModel() -> getShipAt(row);
    QDialog *dialog = getShipDialog(ship);
    dialog -> show();
    dialog -> raise();
  } // WndMain::doEditShip

  void WndMain::doIces() {
    DlgIces::getTheDialog() -> show();
  } // WndMain::doIces

  void WndMain::doNewRamjet() {
    int row;
    Ramjet *ramjet = MdlShip::getTheModel() -> newRamjet(row);
    gotoTableRow(getTblShips(), row);
    QDialog *dialog = getShipDialog(ramjet);
    dialog -> show();
    dialog -> raise();
  } // WndMain::doNewRamjet

  void WndMain::doNewRocket() {
    int row;
    Rocket *rocket = MdlShip::getTheModel() -> newRocket(row);
    gotoTableRow(getTblShips(), row);
    QDialog *dialog = getShipDialog(rocket);
    dialog -> show();
    dialog -> raise();
  } // WndMain::doNewRocket

  void WndMain::doQuit() {
    while (fShipDialogs.count() > 0) {
      QMap <QString, QDialog*>::iterator iter = fShipDialogs.begin();
      QDialog *dialog = *iter;
      dialog -> close();
      fShipDialogs.erase(iter);
    } // while
    close();
  } // WndMain::doQuit

  void WndMain::doReactions() {
    DlgReactions::getTheDialog() -> show();
  } // WndMain::doReactions

  void WndMain::dropShipDialog(Ship *aShip) {
    QString key = aShip -> getID().toString();
    if (!fShipDialogs.contains(key)) {
      return;
    } // if
    QDialog *dialog = fShipDialogs[key];
    fShipDialogs.remove(key);
    delete dialog;
  } // WndMain::dropShipDialog

  QAction* WndMain::getActDeleteShip() {
    if (!fActDeleteShip) {
      fActDeleteShip = new QAction(tr("&Delete Ship"), this);
      fActDeleteShip -> setShortcut(tr("Delete"));
      connect(fActDeleteShip, SIGNAL(triggered()), this, SLOT(doDeleteShip()));
    } // if
    return fActDeleteShip;
  } // WndMain::getActDeleteShip

  QAction* WndMain::getActEditShip() {
    if (!fActEditShip) {
      fActEditShip = new QAction(tr("&Edit Ship"), this);
      fActEditShip -> setShortcut(tr("Return"));
      connect(fActEditShip, SIGNAL(triggered()), this, SLOT(doEditShip()));
    } // if
    return fActEditShip;
  } // WndMain::getActEditShip

  QAction* WndMain::getActIces() {
    if (!fActIces) {
      fActIces = new QAction(tr("&Ice types"), this);
      connect(fActIces, SIGNAL(triggered()), this, SLOT(doIces()));
    } // if
    return fActIces;
  } // WndMain::getActIces

  QAction* WndMain::getActNewRamjet() {
    if (!fActNewRamjet) {
      fActNewRamjet = new QAction(tr("Ram&jet"), this);
      connect(fActNewRamjet, SIGNAL(triggered()), this, SLOT(doNewRamjet()));
    } // if
    return fActNewRamjet;
  } // WndMain::getActNewRamjet

  QAction* WndMain::getActNewRocket() {
    if (!fActNewRocket) {
      fActNewRocket = new QAction(tr("&Rocket"), this);
      connect(fActNewRocket, SIGNAL(triggered()), this, SLOT(doNewRocket()));
    } // if
    return fActNewRocket;
  } // WndMain::getActNewRocket

  QAction* WndMain::getActQuit() {
    if (!fActQuit) {
      fActQuit = new QAction(tr("&Quit"), this);
      fActQuit -> setShortcut(tr("Ctrl+Q"));
      fActQuit -> setStatusTip(tr("Exit program"));
      connect(fActQuit, SIGNAL(triggered()), this, SLOT(close()));
    } // if
    return fActQuit;
  } // WndMain::getActQuit

  QAction* WndMain::getActReactions() {
    if (!fActReactions) {
      fActReactions = new QAction(tr("&Reaction types"), this);
      connect(fActReactions, SIGNAL(triggered()), this, SLOT(doReactions()));
    } // if
    return fActReactions;
  } // WndMain::getActReactions

  QVBoxLayout* WndMain::getBoxMain() {
    if (!fBoxMain) {
      fBoxMain = new QVBoxLayout();
      fBoxMain -> addWidget(getLbShips());
      fBoxMain -> addWidget(getTblShips());
    } // if
    return fBoxMain;
  } // WndMain::getBoxMain

  QAction* WndMain::getCtxActDeleteShip() {
    if (!fCtxActDeleteShip) {
      fCtxActDeleteShip = new QAction(tr("&Delete ship"), this);
      connect(fCtxActDeleteShip, SIGNAL(triggered()), this,
        SLOT(doDeleteShip()));
    } // if
    return fCtxActDeleteShip;
  } // WndMain::getCtxActDeleteShip

  QAction* WndMain::getCtxActEditShip() {
    if (!fCtxActEditShip) {
      fCtxActEditShip = new QAction(tr("&Edit ship"), this);
      connect(fCtxActEditShip, SIGNAL(triggered()), this,
        SLOT(doEditShip()));
    } // if
    return fCtxActEditShip;
  } // WndMain::getCtxActEditShip

  QMenu* WndMain::getCtxMnShips() {
    if (!fCtxMnShips) {
      fCtxMnShips = new QMenu(this);
      fCtxMnShips -> addAction(getCtxActEditShip());
      fCtxMnShips -> addAction(getCtxActDeleteShip());
    } // if
    return fCtxMnShips;
  } // WndMain::getCtxMnShips

  QLabel* WndMain::getLbShips() {
    if (!fLbShips) {
      fLbShips = new QLabel(tr("L&ist of ships:"));
      fLbShips -> setBuddy(getTblShips());
    } // if
    return fLbShips;
  } // WndMain::getLbShips

  QMenu* WndMain::getMnFile() {
    if (!fMnFile) {
      fMnFile = menuBar() -> addMenu(tr("&File"));
      getMnNewShip();
      fMnFile -> addAction(getActEditShip());
      fMnFile -> addAction(getActDeleteShip());
      getSepQuit();
      fMnFile -> addAction(getActQuit());
    } // if
    return fMnFile;
  } // WndMain::getMnFile

  QMenu* WndMain::getMnNewShip() {
    if (!fMnNewShip) {
      fMnNewShip = getMnFile() -> addMenu(tr("&New Ship"));
      fMnNewShip -> addAction(getActNewRocket());
      fMnNewShip -> addAction(getActNewRamjet());
    } // if
    return fMnNewShip;
  } // WndMain::getMnNewShip

  QMenu* WndMain::getMnOptions() {
    if (!fMnOptions) {
      fMnOptions = menuBar() -> addMenu(tr("&Options"));
      fMnOptions -> addAction(getActReactions());
      fMnOptions -> addAction(getActIces());
    } // if
    return fMnOptions;
  } // WndMain::getMnOptions

  QAction* WndMain::getSepQuit() {
    if (!fSepQuit) {
      fSepQuit = getMnFile() -> addSeparator();
    } // if
    return fSepQuit;
  } // QndMain::getSepQuit

  QDialog* WndMain::getShipDialog(Ship *aShip) {
    QString key = aShip -> getID().toString();
    if (fShipDialogs.contains(key)) {
      return fShipDialogs[key];
    } else {
      if (Rocket *rocket = dynamic_cast<Rocket*>(aShip)) {
        DlgRocket *dlgRocket = new DlgRocket();
	dlgRocket -> setRocket(rocket);
	fShipDialogs.insert(key, dlgRocket);
	return dlgRocket;
      } else if (Ramjet *ramjet = dynamic_cast<Ramjet*>(aShip)) {
        DlgRamjet *dlgRamjet = new DlgRamjet();
        dlgRamjet -> setRamjet(ramjet);
	fShipDialogs.insert(key, dlgRamjet);
	return dlgRamjet;
      } // if
    } // if
    return NULL;
  } // WndMain::getShipDialog

  QTableView* WndMain::getTblShips() {
    if (!fTblShips) {
      fTblShips = new QTableView();
      fTblShips -> setModel(MdlShip::getTheModel());
      connect(fTblShips, SIGNAL(doubleClicked(const QModelIndex&)), this,
        SLOT(doEditShip()));
      fTblShips -> setContextMenuPolicy(Qt::CustomContextMenu);
      connect(fTblShips, SIGNAL(customContextMenuRequested(QPoint)), this,
        SLOT(doCtxMnShips(const QPoint&)));
      QHeaderView *vheader = fTblShips -> verticalHeader();
      vheader -> setDefaultSectionSize(vheader -> minimumSectionSize());
    } // if
    return fTblShips;
  } // WndMain::getTblShips

  WndMain* WndMain::getTheWindow() {
    if (!fTheWindow) {
      fTheWindow = new WndMain();
    } // if
    return fTheWindow;
  } // WndMain::getTheWindow

  QWidget* WndMain::getWgtMain() {
    if (!fWgtMain) {
      fWgtMain = new QWidget();
      fWgtMain -> setLayout(getBoxMain());
    } // if
    return fWgtMain;
  } // WndMain::getWgtMain

  void WndMain::putShip(Ship *aShip) {
    MdlShip *model = MdlShip::getTheModel();
    model -> removeShip(aShip);
    int row = model -> addShip(aShip);
    gotoTableRow(getTblShips(), row);
  } // WndMain::putShip

} // GerFuSp
